<?php
defined( 'ABSPATH' ) or die( 'Something went wrong.' );


/**
 * Content of the settings field for the non login time slot.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_non_login_time_slot_field( $args, $instance ) {
	$name_attribute = 'secupress_' . $instance->get_current_module() . '_settings[' . $args['name'] . ']';

	// Value.
	if ( isset( $args['value'] ) ) {
		$value = $args['value'];
	} else {
		$value = secupress_get_module_option( $args['name'] );
	}

	if ( is_null( $value ) ) {
		$value = $args['default'];
	}

	$from_hour   = isset( $value['from_hour'] )   ? (int) $value['from_hour']   : 0;
	$from_minute = isset( $value['from_minute'] ) ? (int) $value['from_minute'] : 0;
	$to_hour     = isset( $value['to_hour'] )     ? (int) $value['to_hour']     : 0;
	$to_minute   = isset( $value['to_minute'] )   ? (int) $value['to_minute']   : 0;

	// Attributes.
	$attributes = ' type="text" class="small-text" size="2" maxlength="2" autocomplete="off"';
	if ( ! empty( $args['attributes'] ) ) {
		foreach ( $args['attributes'] as $attribute => $attribute_value ) {
			$attributes .= ' ' . $attribute . '="' . esc_attr( $attribute_value ) . '"';
		}
	}

	echo $args['label'] ? '<p id="' . $args['name'] . '-time-slot-label">' . $args['label'] . '</p>' : '';
	?>
	<fieldset aria-labelledby="<?php echo $args['name']; ?>-time-slot-label">
		<legend class="screen-reader-text"><?php _e( 'Start hour and minute', 'secupress' ); ?></legend>
		<label>
			<span class="label-before" aria-hidden="true"><?php _ex( 'From', 'starting hour + minute', 'secupress' ); ?></span>
			<span class="screen-reader-text"><?php _e( 'Hour' ); ?></span>
			<input id="<?php echo $args['name']; ?>_from_hour" name="<?php echo $name_attribute; ?>[from_hour]" value="<?php echo str_pad( $from_hour, 2, 0, STR_PAD_LEFT ); ?>" pattern="0?[0-9]|1[0-9]|2[0-3]"<?php echo $attributes; ?>>
			<span aria-hidden="true"><?php _ex( 'h', 'hour', 'secupress' ); ?></span>
		</label>
		<label>
			<span class="screen-reader-text"><?php _e( 'Minute' ); ?></span>
			<input id="<?php echo $args['name']; ?>_from_minute" name="<?php echo $name_attribute; ?>[from_minute]" value="<?php echo str_pad( $from_minute, 2, 0, STR_PAD_LEFT ); ?>" pattern="0?[0-9]|[1-5][0-9]"<?php echo $attributes; ?>>
			<span aria-hidden="true"><?php _ex( 'min', 'minute', 'secupress' ); ?></span>
		</label>
	</fieldset>

	<fieldset aria-labelledby="<?php echo $args['name']; ?>-time-slot-label">
		<legend class="screen-reader-text"><?php _e( 'End hour and minute', 'secupress' ); ?></legend>
		<label>
			<span class="label-before" aria-hidden="true"><?php _ex( 'To', 'ending hour + minute', 'secupress' ) ?></span>
			<span class="screen-reader-text"><?php _e( 'Hour' ); ?></span>
			<input id="<?php echo $args['name']; ?>_to_hour" name="<?php echo $name_attribute; ?>[to_hour]" value="<?php echo str_pad( $to_hour, 2, 0, STR_PAD_LEFT ); ?>" pattern="0?[0-9]|1[0-9]|2[0-3]"<?php echo $attributes; ?>>
			<span aria-hidden="true"><?php _ex( 'h', 'hour', 'secupress' ); ?></span>
		</label>
		<label>
			<span class="screen-reader-text"><?php _e( 'Minute' ); ?></span>
			<input id="<?php echo $args['name']; ?>_to_minute" name="<?php echo $name_attribute; ?>[to_minute]" value="<?php echo str_pad( $to_minute, 2, 0, STR_PAD_LEFT ); ?>" pattern="0?[0-9]|[1-5][0-9]"<?php echo $attributes; ?>>
			<span aria-hidden="true"><?php _ex( 'min', 'minute', 'secupress' ); ?></span>
		</label>
	</fieldset>
	<?php
}


/**
 * Content of the settings field for the countries management.
 *
 * @since 1.0
 * @author Julio Potier
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_countries_field( $args, $instance ) {
	$name_attribute = 'secupress_' . $instance->get_current_module() . '_settings[' . $args['name'] . ']';

	// Value.
	if ( isset( $args['value'] ) ) {
		$value = $args['value'];
	} else {
		$value = secupress_get_module_option( $args['name'] );
	}

	if ( is_null( $value ) ) {
		$value = $args['default'];
	}
	$value = array_flip( array_filter( (array) $value ) );

	// Attributes.
	$attributes = '';
	if ( ! empty( $args['attributes'] ) ) {
		foreach ( $args['attributes'] as $attribute => $attribute_value ) {
			$attributes .= ' ' . $attribute . '="' . esc_attr( $attribute_value ) . '"';
		}
	}
	$disabled_class = ! empty( $args['attributes']['disabled'] ) ? ' disabled' : '';
	$disabled_attr  = $disabled_class ? ' class="disabled"' : '';
	$_countries     = array( 'AF' => array( 0 => 'Africa', 'AO' => 'Angola', 'BF' => 'Burkina Faso', 'BI' => 'Burundi', 'BJ' => 'Benin', 'BW' => 'Botswana', 'CD' => 'Congo, The Democratic Republic of the', 'CF' => 'Central African Republic', 'CG' => 'Congo', 'CI' => 'Cote D’Ivoire', 'CM' => 'Cameroon', 'CV' => 'Cape Verde', 'DJ' => 'Djibouti', 'DZ' => 'Algeria', 'EG' => 'Egypt', 'EH' => 'Western Sahara', 'ER' => 'Eritrea', 'ET' => 'Ethiopia', 'GA' => 'Gabon', 'GH' => 'Ghana', 'GM' => 'Gambia', 'GN' => 'Guinea', 'GQ' => 'Equatorial Guinea', 'GW' => 'Guinea-Bissau', 'KE' => 'Kenya', 'KM' => 'Comoros', 'LR' => 'Liberia', 'LS' => 'Lesotho', 'LY' => 'Libya', 'MA' => 'Morocco', 'MG' => 'Madagascar', 'ML' => 'Mali', 'MR' => 'Mauritania', 'MU' => 'Mauritius', 'MW' => 'Malawi', 'MZ' => 'Mozambique', 'NA' => 'Namibia', 'NE' => 'Niger', 'NG' => 'Nigeria', 'RE' => 'Reunion', 'RW' => 'Rwanda', 'SC' => 'Seychelles', 'SD' => 'Sudan', 'SH' => 'Saint Helena', 'SL' => 'Sierra Leone', 'SN' => 'Senegal', 'SO' => 'Somalia', 'ST' => 'Sao Tome and Principe', 'SZ' => 'Swaziland', 'TD' => 'Chad', 'TG' => 'Togo', 'TN' => 'Tunisia', 'TZ' => 'Tanzania, United Republic of', 'UG' => 'Uganda', 'YT' => 'Mayotte', 'ZA' => 'South Africa', 'ZM' => 'Zambia', 'ZW' => 'Zimbabwe', 'SS' => 'South Sudan' ), 'AN' => array( 0 => 'Antarctica', 'AQ' => 'Antarctica', 'BV' => 'Bouvet Island', 'GS' => 'South Georgia and the South Sandwich Islands', 'HM' => 'Heard Island and McDonald Islands', 'TF' => 'French Southern Territories' ), 'AS' => array( 0 => 'Asia', 'AP' => 'Asia/Pacific Region', 'AE' => 'United Arab Emirates', 'AF' => 'Afghanistan', 'AM' => 'Armenia', 'AZ' => 'Azerbaijan', 'BD' => 'Bangladesh', 'BH' => 'Bahrain', 'BN' => 'Brunei Darussalam', 'BT' => 'Bhutan', 'CC' => 'Cocos (Keeling) Islands', 'CN' => 'China', 'CX' => 'Christmas Island', 'CY' => 'Cyprus', 'GE' => 'Georgia', 'HK' => 'Hong Kong', 'ID' => 'Indonesia', 'IL' => 'Israel', 'IN' => 'India', 'IO' => 'British Indian Ocean Territory', 'IQ' => 'Iraq', 'IR' => 'Iran, Islamic Republic of', 'JO' => 'Jordan', 'JP' => 'Japan', 'KG' => 'Kyrgyzstan', 'KH' => 'Cambodia', 'KP' => 'Korea, Democratic People’s Republic of', 'KR' => 'Korea, Republic of', 'KW' => 'Kuwait', 'KZ' => 'Kazakhstan', 'LA' => 'Lao People’s Democratic Republic', 'LB' => 'Lebanon', 'LK' => 'Sri Lanka', 'MM' => 'Myanmar', 'MN' => 'Mongolia', 'MO' => 'Macau', 'MV' => 'Maldives', 'MY' => 'Malaysia', 'NP' => 'Nepal', 'OM' => 'Oman', 'PH' => 'Philippines', 'PK' => 'Pakistan', 'PS' => 'Palestinian Territory', 'QA' => 'Qatar', 'SA' => 'Saudi Arabia', 'SG' => 'Singapore', 'SY' => 'Syrian Arab Republic', 'TH' => 'Thailand', 'TJ' => 'Tajikistan', 'TM' => 'Turkmenistan', 'TL' => 'Timor-Leste', 'TW' => 'Taiwan', 'UZ' => 'Uzbekistan', 'VN' => 'Vietnam', 'YE' => 'Yemen' ), 'EU' => array( 0 => 'Europe', 'AD' => 'Andorra', 'AL' => 'Albania', 'AT' => 'Austria', 'BA' => 'Bosnia and Herzegovina', 'BE' => 'Belgium', 'BG' => 'Bulgaria', 'BY' => 'Belarus', 'CH' => 'Switzerland', 'CZ' => 'Czech Republic', 'DE' => 'Germany', 'DK' => 'Denmark', 'EE' => 'Estonia', 'ES' => 'Spain', 'FI' => 'Finland', 'FO' => 'Faroe Islands', 'FR' => 'France', 'GB' => 'United Kingdom', 'GI' => 'Gibraltar', 'GR' => 'Greece', 'HR' => 'Croatia', 'HU' => 'Hungary', 'IE' => 'Ireland', 'IS' => 'Iceland', 'IT' => 'Italy', 'LI' => 'Liechtenstein', 'LT' => 'Lithuania', 'LU' => 'Luxembourg', 'LV' => 'Latvia', 'MC' => 'Monaco', 'MD' => 'Moldova, Republic of', 'MK' => 'Macedonia', 'MT' => 'Malta', 'NL' => 'Netherlands', 'NO' => 'Norway', 'PL' => 'Poland', 'PT' => 'Portugal', 'RO' => 'Romania', 'RU' => 'Russian Federation', 'SE' => 'Sweden', 'SI' => 'Slovenia', 'SJ' => 'Svalbard and Jan Mayen', 'SK' => 'Slovakia', 'SM' => 'San Marino', 'TR' => 'Turkey', 'UA' => 'Ukraine', 'VA' => 'Holy See (Vatican City State)', 'RS' => 'Serbia', 'ME' => 'Montenegro', 'AX' => 'Aland Islands', 'GG' => 'Guernsey', 'IM' => 'Isle of Man', 'JE' => 'Jersey' ), 'OC' => array( 0 => 'Oceania', 'AS' => 'American Samoa', 'AU' => 'Australia', 'CK' => 'Cook Islands', 'FJ' => 'Fiji', 'FM' => 'Micronesia, Federated States of', 'GU' => 'Guam', 'KI' => 'Kiribati', 'MH' => 'Marshall Islands', 'MP' => 'Northern Mariana Islands', 'NC' => 'New Caledonia', 'NF' => 'Norfolk Island', 'NR' => 'Nauru', 'NU' => 'Niue', 'NZ' => 'New Zealand', 'PF' => 'French Polynesia', 'PG' => 'Papua New Guinea', 'PN' => 'Pitcairn Islands', 'PW' => 'Palau', 'SB' => 'Solomon Islands', 'TK' => 'Tokelau', 'TO' => 'Tonga', 'TV' => 'Tuvalu', 'UM' => 'United States Minor Outlying Islands', 'VU' => 'Vanuatu', 'WF' => 'Wallis and Futuna', 'WS' => 'Samoa' ), 'NA' => array( 0 => 'North America', 'AG' => 'Antigua and Barbuda', 'AI' => 'Anguilla', 'CW' => 'Curacao', 'AW' => 'Aruba', 'BB' => 'Barbados', 'BM' => 'Bermuda', 'BS' => 'Bahamas', 'BZ' => 'Belize', 'CA' => 'Canada', 'CR' => 'Costa Rica', 'CU' => 'Cuba', 'DM' => 'Dominica', 'DO' => 'Dominican Republic', 'SX' => 'Sint Maarten (Dutch part)', 'GD' => 'Grenada', 'GL' => 'Greenland', 'GP' => 'Guadeloupe', 'GT' => 'Guatemala', 'HN' => 'Honduras', 'HT' => 'Haiti', 'JM' => 'Jamaica', 'KN' => 'Saint Kitts and Nevis', 'KY' => 'Cayman Islands', 'LC' => 'Saint Lucia', 'MQ' => 'Martinique', 'MS' => 'Montserrat', 'MX' => 'Mexico', 'NI' => 'Nicaragua', 'PA' => 'Panama', 'PM' => 'Saint Pierre and Miquelon', 'PR' => 'Puerto Rico', 'SV' => 'El Salvador', 'TC' => 'Turks and Caicos Islands', 'TT' => 'Trinidad and Tobago', 'US' => 'United States', 'VC' => 'Saint Vincent and the Grenadines', 'VG' => 'Virgin Islands, British', 'VI' => 'Virgin Islands, U.S.', 'BL' => 'Saint Barthelemy', 'MF' => 'Saint Martin', 'BQ' => 'Bonaire, Saint Eustatius and Saba' ), 'SA' => array( 0 => 'South America', 'AR' => 'Argentina', 'BO' => 'Bolivia', 'BR' => 'Brazil', 'CL' => 'Chile', 'CO' => 'Colombia', 'EC' => 'Ecuador', 'FK' => 'Falkland Islands (Malvinas)', 'GF' => 'French Guiana', 'GY' => 'Guyana', 'PE' => 'Peru', 'PY' => 'Paraguay', 'SR' => 'Suriname', 'UY' => 'Uruguay', 'VE' => 'Venezuela' ) );

	foreach ( $_countries as $code_country => $countries ) {
		$title   = array_shift( $countries );
		$checked = array_intersect_key( $value, $countries );
		$checked = ! empty( $checked );
		?>
		<label class="continent<?php echo $disabled_class; ?>">
			<input type="checkbox" value="continent-<?php echo $code_country; ?>"<?php checked( $checked ); ?><?php echo $attributes; ?>>
			<?php echo '<span class="label-text">' . $title . '</span>'; ?>
		</label>
		<button type="button" class="hide-if-no-js expand_country"><img src="data:image/gif;base64,R0lGODlhEAAQAMQAAAAAAM/Iu3iYtcK4qPX18bDC09/b0ubm5v///9jTye3t59LMv8a+ruXh2tzYz/j4+PDw7NbRxuTh2f///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAUUABMALAAAAAAQABAAAAVI4CSOZGmeaKqubFkIcCwUp4DcOCLUOHA/O5PgQQQ8II1gSUAAOJ0GJUkAgSgAB4lDOhJoE4DIIsAVCRaMgVpdnrxkMFprjgoBADs=" alt="+" title="<?php esc_attr_x( 'Expand', 'verb', 'secupress' ); ?>" /></button>
		<fieldset class="hide-if-js">
			<legend class="screen-reader-text"><span><?php echo $title; ?></span></legend>
			<?php
			foreach ( $countries as $code => $title ) {
				$args['label_for'] = $args['name'] . '_' . $code;
				?>
				<div>
					<span class="secupress-tree-dash"></span>
					<label<?php echo $disabled_attr; ?>>
						<input type="checkbox" id="<?php echo $args['label_for']; ?>" name="<?php echo $name_attribute; ?>[]" value="<?php echo $code; ?>"<?php checked( isset( $value[ $code ] ) ); ?> data-code-country="<?php echo $code_country; ?>"<?php echo $attributes; ?>>
						<?php echo '<span class="label-text">' . $title . '</span>'; ?>
					</label>
				</div>
				<?php
			}
			?>
		</fieldset>
		<br/>
		<?php
	}
}


/**
 * Content of the settings field for the file scanner.
 *
 * @since 2.5 Revamp.
 * @since 1.0
 * @author Julio Potier
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_file_scanner_field( $args, $instance ) {
	global $running, $scan_ko;
	
	if ( $running ) {
		secupress_print_scanner_ui();
		return;
	}

	$files = secupress_file_scanner_get_result();

	if ( false === $files ) {
		secupress_print_scanner_ui();
		echo '<br>';
		secupress_display_malware_scanners_ko( 1 );
		secupress_display_expert_update_button();
		return;
	}

	$nothing_found = true;
	$scanner_results = [];
	
	/**
	 * Files that are not part of the WordPress installation.
	 */
	if ( ! empty( $files['not-wp-files'] ) ) {
		$nothing_found          = false;
		$expert_mode            = isset( $_GET['sp-deletemode'] ) && $_GET['sp-deletemode'] === '1';
		
		// Split files into 3 categories: plugins, themes, and others
		$plugins_path = trailingslashit( str_replace( ABSPATH, '', WP_PLUGIN_DIR ) );
		$themes_path  = trailingslashit( str_replace( ABSPATH, '', WP_CONTENT_DIR . '/themes' ) );////
		
		$files_plugins = array_filter( $files['not-wp-files'], function( $file ) use ( $plugins_path ) {
			return strpos( $file, $plugins_path ) === 0;
		} );
		
		$files_themes = array_filter( $files['not-wp-files'], function( $file ) use ( $themes_path ) {
			return strpos( $file, $themes_path ) === 0;
		} );
		
		$files_others = array_filter( $files['not-wp-files'], function( $file ) use ( $plugins_path, $themes_path ) {
			return strpos( $file, $plugins_path ) !== 0 && strpos( $file, $themes_path ) !== 0;
		} );
		
		// Define the 3 categories with their titles
		$file_categories = [
			'others'  => [
				'files' => $files_others,
				'title' => __( 'Suspicious Files Found in WordPress Directories', 'secupress' ),
				'id'    => 'others',
			],
			'plugins' => [
				'files' => $files_plugins,
				'title' => __( 'Suspicious Files Found in Plugins Directories', 'secupress' ),
				'id'    => 'plugins',
			],
			'themes'  => [
				'files' => $files_themes,
				'title' => __( 'Suspicious Files Found in Themes Directories', 'secupress' ),
				'id'    => 'themes',
			],
		];
		
		ob_start();
		?>
		<form id="form-not-wp-files" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_action_on_scanned_files' ), 'secupress_action_on_scanned_files' ) ); ?>" method="post">

			<?php
			$table_index = 0;
			$counter     = 0;
			foreach ( $file_categories as $category_key => $category ) :
				if ( empty( $category['files'] ) ) {
					continue;
				}
				$table_index++;
			?>
			<h3 class="secupress-section-title"><?php echo esc_html( $category['title'] ); ?></h3>

			<table class="wp-list-table widefat fixed striped posts secupress-not-wp-files-table secupress-malware-files-table secupress-check-group">
				<thead>
					<tr>
						<?php if ( $expert_mode ) { ?>
						<td id="cb-<?php echo $category['id']; ?>" class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-<?php echo $category['id']; ?>-1"><?php _e( 'Select All', 'secupress-pro' ); ?></label>
							<input id="cb-select-all-<?php echo $category['id']; ?>-1" type="checkbox" class="secupress-toggle-check">
						</td>
						<?php } ?>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress-pro' ); ?></th>
						<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-size"><?php _e( 'Size', 'secupress-pro' ); ?></th>
						<th scope="col" class="manage-column column-modified"><?php _e( 'Last Modified', 'secupress-pro' ); ?></th>
					</tr>
				</thead>
				<tbody id="the-list-<?php echo $category['id']; ?>">
					<?php
					foreach ( $category['files'] as $diff_file ) {
						$file_path      = ABSPATH . $diff_file;
						$file_size      = file_exists( $file_path ) ? size_format( filesize( $file_path ), 2 ) : '-';
						$file_mtime     = file_exists( $file_path ) ? filemtime( $file_path ) : 0;
						$file_modified  = $file_mtime ? human_time_diff( $file_mtime ) . ' ' . __( 'ago', 'secupress-pro' ) : '-';
						$file_full_date = $file_mtime ? date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $file_mtime ) : '';
						$row_id         = sanitize_html_class( $diff_file );
						?>
						<tr id="file-<?php echo $row_id; ?>" class="secupress-not-wp-file-row">
							<?php if ( $expert_mode ) { ?>
							<th scope="row" class="check-column">
								<label class="screen-reader-text" for="cb-select-<?php echo $row_id; ?>"><?php printf( __( 'Select %s', 'secupress-pro' ), $diff_file ); ?></label>
								<input id="cb-select-<?php echo $row_id; ?>" type="checkbox" name="files[]" value="<?php echo esc_attr( $diff_file ); ?>" class="secupress-row-check">
							</th>
							<?php } ?>
							<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'File', 'secupress-pro' ); ?>">
								<strong><code>/<?php echo esc_html( $diff_file ); ?></code></strong>
								<?php if ( ! $expert_mode ) { ?>
								<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress-pro' ); ?></span></button>
								<?php } ?>
							</td>
							<td class="column-signatures" data-colname="<?php esc_attr_e( 'Signatures', 'secupress' ); ?>">
								<?php
								$signatures = secupress_check_malware( $diff_file, false, true );
								$has_signatures = is_array( $signatures ) && ! empty( $signatures );
								if ( $has_signatures ) {
									$counter++;
									?>
									<span class="secupress-signature-count" title="<?php echo esc_attr( implode( ', ', $signatures ) ); ?>">
										<span class="dashicons dashicons-warning" style="color:#d63638;"></span>
										<?php printf( _n( '%d signature', '%d signatures', count( $signatures ), 'secupress' ), count( $signatures ) ); ?>
									</span>
									<button type="button" class="button-link secupress-toggle-signatures" data-target="signatures-<?php echo $row_id; ?>">
										<span class="dashicons dashicons-arrow-down-alt2"></span>
									</button>
									<div id="signatures-<?php echo $row_id; ?>" class="secupress-signatures-list" style="display:none;">
										<?php
										$sig_codes = array_map( function( $sig ) {
											return '<code>' . $sig . '</code>';
										}, $signatures );
										echo implode( ', ', $sig_codes );
										?>
									</div>
								<?php } else { ?>
									<span class="dashicons dashicons-yes-alt" style="color:#00a32a;"></span>
									<?php _e( 'Clean', 'secupress' ); ?>
								<?php } ?>
							</td>
							<td class="column-size" data-colname="<?php esc_attr_e( 'Size', 'secupress-pro' ); ?>">
								<?php echo esc_html( $file_size ); ?>
							</td>
							<td class="column-modified" data-colname="<?php esc_attr_e( 'Last Modified', 'secupress-pro' ); ?>">
								<?php if ( $file_full_date ) { ?>
									<abbr title="<?php echo esc_attr( $file_full_date ); ?>"><?php echo esc_html( $file_modified ); ?></abbr>
								<?php } else { ?>
									<?php echo esc_html( $file_modified ); ?>
								<?php } ?>
							</td>
						</tr>
					<?php } ?>
				</tbody>
				<tfoot>
					<tr>
						<?php if ( $expert_mode ) { ?>
						<td class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-<?php echo $category['id']; ?>-2"><?php _e( 'Select All', 'secupress-pro' ); ?></label>
							<input id="cb-select-all-<?php echo $category['id']; ?>-2" type="checkbox" class="secupress-toggle-check">
						</td>
						<?php } ?>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress-pro' ); ?></th>
						<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-size"><?php _e( 'Size', 'secupress-pro' ); ?></th>
						<th scope="col" class="manage-column column-modified"><?php _e( 'Last Modified', 'secupress-pro' ); ?></th>
					</tr>
				</tfoot>
			</table>
			<?php endforeach; ?>

			<?php if ( $expert_mode ) { ?>
				<p class="submit secupress-clearfix">
					<?php submit_button( __( 'Delete selected files', 'secupress-pro' ), 'secupress-button secupress-button-mini alignright', 'submit-delete-files', false ); ?>
				</p>
			<?php } elseif ( secupress_is_expert_mode() ) { ?>
				<p class="submit secupress-clearfix secupress-show-expert">
					<a href="<?php echo esc_url( add_query_arg( 'sp-deletemode', '1' ) ); ?>" class="secupress-button secupress-button-mini alignright"><?php _e( 'Switch to delete mode', 'secupress-pro' ); ?></a>
				</p>
			<?php } ?>
		</form>
		<?php
		$scanner_results['malware_keywords']['html'] = ob_get_clean();
		$scanner_results['malware_keywords']['count'] = $counter;
	}

	/**
	 * WordPress Core Files Integrity results (missing + modified).
	 */
	$core_integrity_html = '';
	$counter = 0;

	// Missing files from WP Core.
	if ( ! empty( $files['missing-wp-files'] ) ) {
		$nothing_found       = false;
		$filelist_select_all = count( $files['missing-wp-files'] ) > 1;

		ob_start();
		?>
		<form id="form-recover-missing-files" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_recover_missing_files' ), 'secupress_recover_missing_files' ) ); ?>" method="post">

			<h3 class="secupress-section-title"><?php _e( 'The followings are missing from WordPress core files', 'secupress' ); ?></h3>

			<table class="wp-list-table widefat fixed striped posts secupress-missing-files-table secupress-check-group">
				<thead>
					<tr>
						<td id="cb-missing-files" class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-missing-files-1"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-missing-files-1" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress' ); ?></th>
					</tr>
				</thead>
				<tbody id="the-list-missing-files">
					<?php
					foreach ( $files['missing-wp-files'] as $miss_file ) {
						$counter++;
						$row_id = sanitize_html_class( $miss_file );
						?>
						<tr id="missing-file-<?php echo $row_id; ?>" class="secupress-missing-file-row">
							<th scope="row" class="check-column">
								<label class="screen-reader-text" for="cb-select-missing-<?php echo $row_id; ?>"><?php printf( __( 'Select %s', 'secupress' ), esc_html( $miss_file ) ); ?></label>
								<input id="cb-select-missing-<?php echo $row_id; ?>" type="checkbox" name="files[]" value="<?php echo esc_attr( $miss_file ); ?>" class="secupress-row-check">
							</th>
							<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'File', 'secupress' ); ?>">
								<strong><code><?php echo esc_html( $miss_file ); ?></code></strong>
								<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
							</td>
						</tr>
						<?php
					}
					?>
				</tbody>
				<tfoot>
					<tr>
						<td class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-missing-files-2"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-missing-files-2" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress' ); ?></th>
					</tr>
				</tfoot>
			</table>

			<p class="submit secupress-clearfix">
				<?php submit_button( __( 'Recover selected files', 'secupress' ), 'secondary alignright secupress-button secupress-button-mini', 'submit-recover-missing-files', false ); ?>
			</p>

		</form>
		<?php
		$core_integrity_html .= ob_get_clean();
	}

	// Modified WP Core files.
	if ( ! empty( $files['modified-wp-files'] ) ) {
		$nothing_found          = false;
		$filelist_li            = '';
		$filelist_select_all    = count( $files['modified-wp-files'] ) > 1;
		$possible_malware_found = false;

		foreach ( $files['modified-wp-files'] as $mod_file ) {

			$malware = secupress_check_malware( $mod_file );
			if ( $malware ) {
				// Display this message if there is at least one malware.
				$possible_malware_found = true;
			}

			$filelist_li .= sprintf(
				'<li class="secupress-files-list-item"><input id="mod-file-%1$s" class="secupress-checkbox secupress-row-check" type="checkbox" name="files[]" value="%3$s" title="%3$s"> <label for="mod-file-%1$s" title="%3$s">%2$s</label> <a target="_blank" href="%4$s" class="secupress-button secupress-button-ghost secupress-button-mini secupress-button-primary"><span class="icon"><i class="dashicons dashicons-plus" aria-hidden="true"></i></span><span class="text">' . __( 'See differences', 'secupress' ) . '</span></a>%5$s</li>',
				sanitize_html_class( $mod_file ),
				esc_html( $mod_file ),
				esc_attr( $mod_file ),
				esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_diff_file&file=' . $mod_file ), 'secupress_diff_file-' . $mod_file ) ),
				$malware // Do not escape.
			);
		}

		ob_start();
		?>
		<form id="form-recover-diff-files" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_action_on_scanned_files' ), 'secupress_action_on_scanned_files' ) ); ?>" method="post">

			<h3 class="secupress-section-title"><?php _e( 'The followings are modified WordPress core files', 'secupress' ); ?></h3>

			<?php if ( $possible_malware_found ) { ?>
				<p><span class="secupress-inline-alert"><?php _e( 'Possible malware found', 'secupress' ); ?></span></p>
			<?php } ?>

			<table class="wp-list-table widefat fixed striped posts secupress-modified-files-table secupress-malware-files-table secupress-check-group">
				<thead>
					<tr>
						<td id="cb-modified-files" class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-modified-files-1"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-modified-files-1" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-actions"><?php _e( 'Actions', 'secupress' ); ?></th>
					</tr>
				</thead>
				<tbody id="the-list-modified-files">
					<?php
					foreach ( $files['modified-wp-files'] as $mod_file ) {
						$counter++;
						$signatures = secupress_check_malware( $mod_file, false, true );
						$row_id  = sanitize_html_class( $mod_file );
						$has_signatures = is_array( $signatures ) && ! empty( $signatures );
						?>
						<tr id="modified-file-<?php echo $row_id; ?>" class="secupress-modified-file-row">
							<th scope="row" class="check-column">
								<label class="screen-reader-text" for="cb-select-modified-<?php echo $row_id; ?>"><?php printf( __( 'Select %s', 'secupress' ), esc_html( $mod_file ) ); ?></label>
								<input id="cb-select-modified-<?php echo $row_id; ?>" type="checkbox" name="files[]" value="<?php echo esc_attr( $mod_file ); ?>" class="secupress-row-check">
							</th>
							<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'File', 'secupress' ); ?>">
								<strong><code><?php echo esc_html( $mod_file ); ?></code></strong>
								<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
							</td>
							<td class="column-signatures" data-colname="<?php esc_attr_e( 'Signatures', 'secupress' ); ?>">
								<?php if ( $has_signatures ) {
									$counter++;
									?>
									<span class="secupress-signature-count" title="<?php echo esc_attr( implode( ', ', $signatures ) ); ?>">
										<span class="dashicons dashicons-warning" style="color:#d63638;"></span>
										<?php printf( _n( '%d signature', '%d signatures', count( $signatures ), 'secupress' ), count( $signatures ) ); ?>
									</span>
									<button type="button" class="button-link secupress-toggle-signatures" data-target="signatures-<?php echo $row_id; ?>">
										<span class="dashicons dashicons-arrow-down-alt2"></span>
									</button>
									<div id="signatures-<?php echo $row_id; ?>" class="secupress-signatures-list" style="display:none;">
										<?php
										$sig_codes = array_map( function( $sig ) {
											return '<code>' . esc_html( $sig ) . '</code>';
										}, $signatures );
										echo implode( ', ', $sig_codes );
										?>
									</div>
								<?php } else { ?>
									<?php echo '—'; ?>
								<?php } ?>
							</td>
							<td class="column-actions" data-colname="<?php esc_attr_e( 'Actions', 'secupress' ); ?>">
								<a target="_blank" href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_diff_file&file=' . $mod_file ), 'secupress_diff_file-' . $mod_file ) ); ?>" class="">
									<span class="text"><?php _e( 'See differences', 'secupress' ); ?></span>
								</a>
							</td>
						</tr>
						<?php
					}
					?>
				</tbody>
				<tfoot>
					<tr>
						<td class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-modified-files-2"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-modified-files-2" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'File', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-actions"><?php _e( 'Actions', 'secupress' ); ?></th>
					</tr>
				</tfoot>
			</table>

			<p class="submit secupress-clearfix">
				<?php submit_button( __( 'Recover selected files', 'secupress' ), 'secondary alignright secupress-button secupress-button-mini', 'submit-recover-diff-files', false ); ?>
			</p>

		</form>
		<?php
		$core_integrity_html .= ob_get_clean();
	}

	if ( ! empty( $core_integrity_html ) ) {
		$scanner_results['core-integrity']['html'] = $core_integrity_html;
		$scanner_results['core-integrity']['count'] = $counter;
	}

	/**
	 * Content Spam
	 */
	$content_spam = [];
	if ( isset( $files['content-spam'] ) && is_array( $files['content-spam'] ) ) {
		$content_spam = array_filter( $files['content-spam'] );
	}

	if ( ! empty( $content_spam ) ) {
		$nothing_found          = false;
		$counter                = 0;
		$filelist_li            = '';
		$prev                   = get_site_option( SECUPRESS_CONTENT_ALLOWED, '' );
		if ( is_array( $prev ) ) {
			$prev = str_rot13( json_encode( $prev ) );
			update_site_option( SECUPRESS_CONTENT_ALLOWED, $prev );
		}
		if ( ! empty( $prev ) ) {
			$prev = json_decode( str_rot13( $prev ), true );
		}
		if ( ! is_array( $prev ) ) {
			$prev = [];
		}
		if ( ! empty( $prev ) ) {
			$prev = array_flip( $prev );
		}

		// Grouper les liens par URL source.
		$grouped_by_url = [];
		foreach ( $content_spam as $tag => $links ) {
			list( $_tag, $_attr ) = explode( '|', $tag );
			foreach ( $links as $id => $link ) {
				if ( isset( $prev[ $id ] ) ) {
					continue;
				}
				$counter++;
				if ( ! isset( $grouped_by_url[ $link ] ) ) {
					$grouped_by_url[ $link ] = [];
				}
				$grouped_by_url[ $link ][] = [
					'tag'   => $tag,
					'_tag'  => $_tag,
					'_attr' => $_attr,
					'link'  => $id,
				];
			}
		}
	
		$filelist_select_all = count( $grouped_by_url ) > 1;
		ob_start();
		if ( $counter ) {
		?>
		<form id="form-check-content-spam" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_action_on_content_spam&act=add' ), 'secupress_action_on_content_spam_add' ) ); ?>" method="post">

			<h3 class="secupress-section-title"><?php _e( 'The following contents point out from your website, please check everything that you allow and fix the rest.', 'secupress' ); ?></h3>

			<table class="wp-list-table widefat fixed striped posts secupress-seo-poisoning-table secupress-check-group">
				<thead>
					<tr>
						<td id="cb-seo-poisoning" class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-seo-poisoning-1"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-seo-poisoning-1" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'Source URL', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-link"><?php _e( 'Link', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-tag"><?php _e( 'Tag / Attribute', 'secupress' ); ?></th>
					</tr>
				</thead>
				<tbody id="the-list-seo-poisoning">
					<?php
					foreach ( $grouped_by_url as $source_url => $found_links ) {
						foreach ( $found_links as $item ) {
							$row_id = sanitize_html_class( $item['link'] );
							?>
							<tr id="seo-poisoning-<?php echo $row_id; ?>" class="secupress-seo-poisoning-row">
								<th scope="row" class="check-column">
									<label class="screen-reader-text" for="cb-select-<?php echo $row_id; ?>"><?php printf( __( 'Select %s', 'secupress' ), esc_html( $item['link'] ) ); ?></label>
									<input id="cb-select-<?php echo $row_id; ?>" type="checkbox" name="content_spam_to_check[]" value="<?php echo esc_attr( $item['link'] ); ?>" class="secupress-row-check" title="<?php printf( __( 'Found in %s', 'secupress' ), esc_attr( $source_url ) ); ?>">
								</th>
								<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'Source URL', 'secupress' ); ?>">
									<strong><?php echo esc_html( $source_url ); ?></strong>
									<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
								</td>
								<td class="column-link" data-colname="<?php esc_attr_e( 'Link', 'secupress' ); ?>">
									<strong><?php echo esc_html( $item['link'] ); ?></strong>
								</td>
								<td class="column-tag" data-colname="<?php esc_attr_e( 'Tag / Attribute', 'secupress' ); ?>">
									<code><?php echo esc_html( "<{$item['_tag']} {$item['_attr']}=\"" . $item['link'] . "\">" ); ?></code>
								</td>
							</tr>
							<?php
						}
					}
					?>
				</tbody>
				<tfoot>
					<tr>
						<td class="manage-column column-cb check-column">
							<label class="screen-reader-text" for="cb-select-all-seo-poisoning-2"><?php _e( 'Select All', 'secupress' ); ?></label>
							<input id="cb-select-all-seo-poisoning-2" type="checkbox" class="secupress-toggle-check">
						</td>
						<th scope="col" class="manage-column column-primary"><?php _e( 'Source URL', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-link"><?php _e( 'Link', 'secupress' ); ?></th>
						<th scope="col" class="manage-column column-tag"><?php _e( 'Tag / Attribute', 'secupress' ); ?></th>
					</tr>
				</tfoot>
			</table>

			<p class="submit secupress-clearfix">
				<?php submit_button( __( 'Add to allowed list', 'secupress' ), 'secondary alignright secupress-button secupress-button-mini', 'submit-allowed-list', false ); ?>
			</p>

		</form>
		<?php
		}

		if ( ! empty( $prev ) ) {
			$prev                = array_flip( $prev );
			$filelist_li         = '';
			$filelist_select_all = count( $prev ) > 1;
			foreach ( $prev as $link ) {
				$filelist_li .= sprintf(
					'<li class="secupress-files-list-item"><input id="content-spam-%1$s" class="secupress-checkbox secupress-row-check" type="checkbox" name="content_spam_to_check[]" value="%3$s"> <label for="content-spam-%1$s" title="%3$s">%2$s</label></li>',
					sanitize_html_class( $link ),
					'<strong>' . esc_html( $link ) . '</strong>',
					esc_attr( $link )
				);
			}
			?>
			<form id="form-allowed-content-spam" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_action_on_content_spam&act=remove' ), 'secupress_action_on_content_spam_remove' ) ); ?>" method="post">

				<h3 class="secupress-section-title"><?php _e( 'The following contents have been allowed earlier. Feel free to remove them from the allowed list.', 'secupress' ); ?></h3>

				<table class="wp-list-table widefat fixed striped posts secupress-seo-poisoning-allowed-table secupress-check-group">
					<thead>
						<tr>
							<td id="cb-seo-poisoning-allowed" class="manage-column column-cb check-column">
								<label class="screen-reader-text" for="cb-select-all-seo-poisoning-allowed-1"><?php _e( 'Select All', 'secupress' ); ?></label>
								<input id="cb-select-all-seo-poisoning-allowed-1" type="checkbox" class="secupress-toggle-check">
							</td>
							<th scope="col" class="manage-column column-primary"><?php _e( 'Allowed Link', 'secupress' ); ?></th>
						</tr>
					</thead>
					<tbody id="the-list-seo-poisoning-allowed">
						<?php
						foreach ( $prev as $link ) {
							$row_id = sanitize_html_class( $link );
							?>
							<tr id="seo-poisoning-allowed-<?php echo $row_id; ?>" class="secupress-seo-poisoning-allowed-row">
								<th scope="row" class="check-column">
									<label class="screen-reader-text" for="cb-select-allowed-<?php echo $row_id; ?>"><?php printf( __( 'Select %s', 'secupress' ), esc_html( $link ) ); ?></label>
									<input id="cb-select-allowed-<?php echo $row_id; ?>" type="checkbox" name="content_spam_to_check[]" value="<?php echo esc_attr( $link ); ?>" class="secupress-row-check">
								</th>
								<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'Allowed Link', 'secupress' ); ?>">
									<strong><?php echo esc_html( $link ); ?></strong>
									<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
								</td>
							</tr>
							<?php
						}
						?>
					</tbody>
					<tfoot>
						<tr>
							<td class="manage-column column-cb check-column">
								<label class="screen-reader-text" for="cb-select-all-seo-poisoning-allowed-2"><?php _e( 'Select All', 'secupress' ); ?></label>
								<input id="cb-select-all-seo-poisoning-allowed-2" type="checkbox" class="secupress-toggle-check">
							</td>
							<th scope="col" class="manage-column column-primary"><?php _e( 'Allowed Link', 'secupress' ); ?></th>
						</tr>
					</tfoot>
				</table>

				<p class="submit secupress-clearfix">
					<?php submit_button( __( 'Remove from allowed list', 'secupress' ), 'secondary alignright secupress-button secupress-button-mini', 'submit-allowed-list', false ); ?>
				</p>

			</form>
			<?php
		}

		$scanner_results['tag_attr']['html'] = ob_get_clean();
		$scanner_results['tag_attr']['count'] = $counter;
	}

	/**
	 * DATABASE Malwares
	 */
	$possible_malware_found = false;
	$dbs = [];
	if ( ! empty( $files['database-wp'] ) ) {
		$dbs = $files['database-wp'];
	}
	$dbs['_opts']  = [];
	// $files['_cpts']  = [];
	$dbs['_posts'] = [];

	// Options
	if ( ! empty( $dbs['opt_val'] ) ) {
		$possible_malware_found = true;
		$nothing_found          = false;
		$dbs['opt_val'] = array_map( function( $item ) {
			return array( $item['option_name'] => $item['option_value']);
		}, $dbs['opt_val'] );
		$dbs['_opts'] = $dbs['opt_val'];
	}
	unset( $dbs['opt_val'] );
	// if ( ! empty( $files['opts'] ) ) {
	// 	$possible_malware_found = true;
	//  $nothing_found          = false;
	// 	$files['opts'] = array_map( function( $item ) {
	// 		return array( $item['option_name'] => $item['option_value']);
	// 	}, $files['opts'] );
	// 	$files['_opts'] = array_merge( $files['_opts'], $files['opts'] );
	// }
	// unset( $files['opts'] );

	// Cpts
	// if ( ! empty( $files['cpts'] ) ) {
	// 	$possible_malware_found = true;
	// 	$files['_cpts'] = array_map( function( $item ) {
	// 		return array( $item['ID'] => $item['post_content']);
	// 	}, $files['cpts'] );
	// }
	// unset( $files['cpts'] );

	// Posts
	if ( ! empty( $dbs['posts'] ) ) {
		$possible_malware_found = true;
		$nothing_found          = false;
		$dbs['_posts'] = array_map( function( $item ) {
			return array( $item['ID'] => $item['post_content']);
		}, $dbs['posts'] );
	}
	unset( $dbs['posts'] );

	if ( $possible_malware_found ) {
		$expert_mode = isset( $_GET['sp-deletemode'] ) && $_GET['sp-deletemode'] === '1';
		ob_start();
		?>
		<h3 class="secupress-section-title"><?php _e( 'Suspicious contents found in your database', 'secupress' ); ?></h3>

		<p><span class="secupress-inline-alert"><?php _e( 'Possible malware found', 'secupress' ); ?></span></p>

		<table class="wp-list-table widefat fixed striped posts secupress-database-scanner-table secupress-check-group">
			<thead>
				<tr>
					<th scope="col" class="manage-column column-type" style="width: 80px;"><?php _e( 'Type', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-primary"><?php _e( 'Name / Title', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-actions"><?php _e( 'Actions', 'secupress' ); ?></th>
				</tr>
			</thead>
			<tbody id="the-list-database">
				<?php
				$keywords = secupress_get_database_malware_keywords();
				$counter = 0;
				foreach ( $dbs['_opts'] as $index => $opts ) {
					foreach ( $opts as $option_name => $option_value ) {
						$row_id = sanitize_html_class( $option_name );
						$found_signatures = [];
						if ( ! empty( $keywords ) && isset( $keywords['posts'] ) && is_array( $keywords['posts'] ) ) {
							foreach ( $keywords['posts'] as $items ) {
								$_count = 0;
								foreach ( $items as $item ) {
									if ( strpos( $item, '%' ) !== false ) {
										$item = stripslashes( $item );
										$item = preg_quote( $item, '/' );
										$item = str_replace( '%', '.*', $item );
										$pattern = '/.*' . $item . '.*/is';
										if ( preg_match( $pattern, $option_value ) ) {
											$_count++;
										}
									} elseif ( ! empty( $item ) && stripos( $option_value, $item ) !== false ) {
										$_count++;
									}
								}
								if ( $_count === count( $items ) ) {
									$found_signatures = $items;
									break;
								}
							}
						}
						$found_signatures = array_unique( $found_signatures );
						$has_signatures = ! empty( $found_signatures );
						
						$delete_link = '';
						if ( $expert_mode && current_user_can( 'manage_options' ) ) {
							$nonce_action = 'secupress_delete_option_' . $option_name;
							$delete_link = sprintf( '<a href="%s">%s</a>', wp_nonce_url( admin_url( 'admin-post.php?action=secupress_delete_option&option_name=' . urlencode( $option_name ) ), $nonce_action ), __( 'Delete', 'secupress' ) );
						}
						$action_display = ! empty( $delete_link ) ? $delete_link : '—';
						?>
						<tr id="db-option-<?php echo $row_id; ?>" class="secupress-database-row">
							<td class="column-type" data-colname="<?php esc_attr_e( 'Type', 'secupress' ); ?>">
								<strong><?php _e( 'Option', 'secupress' ); ?></strong>
							</td>
							<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'Name / Title', 'secupress' ); ?>">
								<code><?php echo esc_html( $option_name ); ?></code>
								<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
							</td>
							<td class="column-signatures" data-colname="<?php esc_attr_e( 'Signatures', 'secupress' ); ?>">
								<?php if ( $has_signatures ) {
									$counter++;
									?>
									<span class="secupress-signature-count" title="<?php echo esc_attr( implode( ', ', $found_signatures ) ); ?>">
										<span class="dashicons dashicons-warning" style="color:#d63638;"></span>
										<?php printf( _n( '%d signature', '%d signatures', count( $found_signatures ), 'secupress' ), count( $found_signatures ) ); ?>
									</span>
									<button type="button" class="button-link secupress-toggle-signatures" data-target="signatures-<?php echo $row_id; ?>">
										<span class="dashicons dashicons-arrow-down-alt2"></span>
									</button>
									<div id="signatures-<?php echo $row_id; ?>" class="secupress-signatures-list" style="display:none;">
										<?php
										$sig_codes = array_map( function( $sig ) {
											return '<code>' . esc_html( $sig ) . '</code>';
										}, $found_signatures );
										echo implode( ', ', $sig_codes );
										?>
									</div>
								<?php } else { ?>
									<span class="dashicons dashicons-yes-alt" style="color:#00a32a;"></span>
									<?php _e( 'Clean', 'secupress' ); ?>
								<?php } ?>
							</td>
							<td class="column-actions" data-colname="<?php esc_attr_e( 'Actions', 'secupress' ); ?>">
								<?php echo $action_display; ?>
							</td>
						</tr>
						<?php
					}
				}
				foreach ( $dbs['_posts'] as $index => $posts ) {
					foreach ( $posts as $ID => $post_content ) {
						$counter++;
						$_post     = get_post( $ID );
						$row_id    = sanitize_html_class( 'post-' . $ID );
						
						$found_signatures = [];
						if ( ! empty( $keywords ) && isset( $keywords['posts'] ) && is_array( $keywords['posts'] ) ) {
							foreach ( $keywords['posts'] as $items ) {
								$_count = 0;
								foreach ( $items as $item ) {
									if ( strpos( $item, '%' ) !== false ) {
										$item = stripslashes( $item );
										$item = preg_quote( $item, '/' );
										$item = str_replace( '%', '.*', $item );
										$pattern = '/.*' . $item . '.*/is';
										if ( preg_match( $pattern, $post_content ) ) {
											$_count++;
										}
									} elseif ( ! empty( $item ) && stripos( $post_content, $item ) !== false ) {
										$_count++;
									}
								}
								if ( $_count === count( $items ) ) {
									$found_signatures = $items;
									break;								}
							}
						}
						$found_signatures = array_unique( $found_signatures );
						$has_signatures = ! empty( $found_signatures );
						
						$edit_link = '';
						$delete_link = '';
						if ( current_user_can( 'edit_post', $ID ) ) {
							$edit_link = sprintf( '<a href="%s">%s</a>', get_edit_post_link( $ID, 'href' ), __( 'Edit', 'secupress' ) );
						}
						if ( $expert_mode && current_user_can( 'delete_post', $ID ) ) {
							$delete_link = sprintf( ' | <a href="%s">%s</a>', get_delete_post_link( $ID, null, true ), __( 'Delete', 'secupress' ) );
						}
						$action_display = ! empty( $edit_link ) ? $edit_link . $delete_link : '—';
						?>
						<tr id="db-post-<?php echo $row_id; ?>" class="secupress-database-row">
							<td class="column-type" data-colname="<?php esc_attr_e( 'Type', 'secupress' ); ?>">
								<strong><?php _e( 'Post', 'secupress' ); ?></strong>
							</td>
							<td class="column-primary has-row-actions" data-colname="<?php esc_attr_e( 'Name / Title', 'secupress' ); ?>">
								<strong><?php echo esc_html( get_the_title( $ID ) ); ?></strong>
								<button type="button" class="toggle-row"><span class="screen-reader-text"><?php _e( 'Show more details', 'secupress' ); ?></span></button>
							</td>
							<td class="column-signatures" data-colname="<?php esc_attr_e( 'Signatures', 'secupress' ); ?>">
								<?php if ( $has_signatures ) { 
									?>
									<span class="secupress-signature-count" title="<?php echo esc_attr( implode( ', ', $found_signatures ) ); ?>">
										<span class="dashicons dashicons-warning" style="color:#d63638;"></span>
										<?php printf( _n( '%d signature', '%d signatures', count( $found_signatures ), 'secupress' ), count( $found_signatures ) ); ?>
									</span>
									<button type="button" class="button-link secupress-toggle-signatures" data-target="signatures-<?php echo $row_id; ?>">
										<span class="dashicons dashicons-arrow-down-alt2"></span>
									</button>
									<div id="signatures-<?php echo $row_id; ?>" class="secupress-signatures-list" style="display:none;">
										<?php
										$sig_codes = array_map( function( $sig ) {
											return '<code>' . esc_html( $sig ) . '</code>';
										}, $found_signatures );
										echo implode( ', ', $sig_codes );
										?>
									</div>
								<?php } else { ?>
									<span class="dashicons dashicons-yes-alt" style="color:#00a32a;"></span>
									<?php _e( 'Clean', 'secupress' ); ?>
								<?php } ?>
							</td>
							<td class="column-actions" data-colname="<?php esc_attr_e( 'Actions', 'secupress' ); ?>">
								<?php echo $action_display; ?>
							</td>
						</tr>
						<?php
					}
				}
				?>
			</tbody>
			<tfoot>
				<tr>
					<th scope="col" class="manage-column column-type" style="width: 80px;"><?php _e( 'Type', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-primary"><?php _e( 'Name / Title', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-signatures"><?php _e( 'Signatures', 'secupress' ); ?></th>
					<th scope="col" class="manage-column column-actions"><?php _e( 'Actions', 'secupress' ); ?></th>
				</tr>
			</tfoot>
		</table>
		<?php if ( $expert_mode ) { ?>
			<p class="submit secupress-clearfix">
				<?php submit_button( __( 'Delete selected items', 'secupress-pro' ), 'secupress-button secupress-button-mini alignright', 'submit-delete-database-items', false ); ?>
			</p>
		<?php } elseif ( secupress_is_expert_mode() ) { ?>
			<p class="submit secupress-clearfix secupress-show-expert">
				<a href="<?php echo esc_url( add_query_arg( 'sp-deletemode', '1' ) ); ?>" class="secupress-button secupress-button-mini alignright"><?php _e( 'Switch to delete mode', 'secupress-pro' ); ?></a>
			</p>
		<?php } ?>
		<?php
		$scanner_results['malware_keywords_db']['html'] = ob_get_clean();
		$scanner_results['malware_keywords_db']['count'] = $counter;
	}

	secupress_print_scanner_ui( $scanner_results );

	
	/**
	 * Old WP files.
	 */
	if ( ! empty( $files['old-wp-files'] ) ) {
		$nothing_found          = false;
		$filelist_li            = '';
		$filelist_select_all    = count( $files['old-wp-files'] ) > 1;
		$possible_malware_found = false;

		foreach ( $files['old-wp-files'] as $old_file ) {

			$malware = secupress_check_malware( $old_file );
			if ( $malware ) {
				// Display this message if there is at least one malware.
				$possible_malware_found = true;
			}

			$filelist_li .= sprintf(
				'<li class="secupress-files-list-item"><input id="old-file-%1$s" class="secupress-checkbox secupress-row-check" type="checkbox" name="files[]" value="%3$s" title="%3$s"> <label for="old-file-%1$s" title="%3$s">%2$s</label>%4$s</li>',
				sanitize_html_class( $old_file ),
				esc_html( $old_file ),
				esc_attr( $old_file ),
				$malware // Do not escape.
			);
		}
		?>
		<form id="form-old-files" action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_old_files' ), 'secupress_old_files' ) ); ?>" method="post">

			<h3 class="secupress-section-title"><?php _e( 'The followings files are old WordPress core ones', 'secupress' ); ?></h3>

			<?php if ( $possible_malware_found ) { ?>
				<p><span class="secupress-inline-alert"><?php _e( 'Possible malware found', 'secupress' ); ?></span></p>
			<?php } ?>

			<fieldset id="secupress-group-old-files" class="secupress-boxed-group small-boxed-group secupress-check-group">

				<?php if ( $filelist_select_all ) { ?>
					<span class="hide-if-no-js"><input id="secupress-toggle-check-old-file-1" type="checkbox" class="secupress-checkbox secupress-toggle-check"> <label for="secupress-toggle-check-old-file-1"><em><?php _e( '(Un)Select All', 'secupress' ); ?></em></label></span>
				<?php } ?>

				<ul class="secupress-files-list">
					<?php echo $filelist_li; ?>
				</ul>

				<?php if ( $filelist_select_all ) { ?>
					<span class="hide-if-no-js"><input id="secupress-toggle-check-old-file-2" type="checkbox" class="secupress-checkbox secupress-toggle-check"> <label for="secupress-toggle-check-old-file-2"><em><?php _e( '(Un)Select All', 'secupress' ); ?></em></label></span>
				<?php } ?>

			</fieldset>

			<p class="submit secupress-clearfix">
				<?php submit_button( __( 'Delete selected files', 'secupress' ), 'secondary alignright secupress-button secupress-button-mini', 'submit-recover-diff-files', false ); ?>
			</p>

		</form>
		<?php
	}
	secupress_display_malware_scanners_ko( $scan_ko );
	secupress_display_expert_update_button();
}

/**
 * Displays a message with doc link if malware scanners are KO
 *
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (int) $scan_ko
 **/
function secupress_display_malware_scanners_ko( $scan_ko ) {
	if( $scan_ko && secupress_is_pro() ) {
		?>
		<p class="description warning secupress-show-expert">
			<span class="dashicons dashicons-warning"></span>
			<strong><?php _e( 'Malware Scanners are unavailable', 'secupress' ); ?></strong><br>
			<?php
			printf( __( 'Please refer to our <a href="%s" target="_blank">documentation</a> on this subject.', 'secupress' ), __( 'https://docs.secupress.me/article/235-malware-scanners', 'secupress' ) );
			?>
		</p>
		<?php
	}
}

/**
 * Displays an update malware data button if expert mode
 *
 * @since 2.6
 * @author Julio Potier
 **/
function secupress_display_expert_update_button() {
	if ( secupress_is_pro() && secupress_is_expert_mode() ) {
		$update_url = wp_nonce_url( admin_url( 'admin-post.php?action=secupress_update_malware_data' ), 'secupress_update_malware_data' );
		?>
		<p class="description warning secupress-show-expert">
			<?php printf( __( 'Version %s', 'secupress' ), secupress_get_option( 'malwaredb_version' ) ); ?><br>
			<?php printf( __( 'Last update: %s', 'secupress' ), secupress_get_option( 'malwaredb_last_update' ) ); ?><br>
			<a id="button-update-malware-data" href="<?php echo esc_url( $update_url ); ?>" class="button button-secondary"><?php _e( 'Update data', 'secupress' ); ?></a>
			<span class="spinner secupress-inline-spinner"></span>
		</p>
		<?php
	}
}

/**
 * Content of the settings field that displays the old backups.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_backup_history_field( $args, $instance ) {
	$backup_files = secupress_get_backup_file_list();
	?>
	<p id="secupress-no-backups"<?php echo $backup_files ? ' class="hidden"' : ''; ?>><em><?php _e( 'No backups found.', 'secupress' ); ?></em></p>

	<form id="form-delete-backups"<?php echo ! $backup_files ? ' class="hidden"' : ''; ?> action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_delete_backups' ), 'secupress_delete_backups' ) ); ?>" method="post">

		<strong id="secupress-available-backups"><?php printf( _n( '%s available Backup', '%s available Backups', count( $backup_files ), 'secupress' ), number_format_i18n( count( $backup_files ) ) ); ?></strong>

		<fieldset class="secupress-boxed-group">
			<legend class="screen-reader-text"><span><?php _e( 'Backups', 'secupress' ); ?></span></legend>
			<?php array_map( 'secupress_print_backup_file_formated', array_reverse( $backup_files ) ); ?>
		</fieldset>

		<p class="submit">
			<button class="secupress-button secupress-button-secondary alignright" type="submit" id="submit-delete-backups">
				<span class="icon">
					<i class="secupress-icon-cross" aria-hidden="true"></i>
				</span>
				<span class="text">
					<?php _e( 'Delete all Backups', 'secupress' ); ?>
				</span>
			</button>
		</p>

	</form>
	<?php
}


/**
 * Content of the settings field that displays the DB tables to backup.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_backup_db_field( $args, $instance ) {
	global $wpdb;

	$wp_tables    = secupress_get_wp_tables();
	$other_tables = secupress_get_non_wp_tables();
	?>
	<form action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_backup_db' ), 'secupress_backup_db' ) ); ?>" id="form-do-db-backup" method="post">

		<fieldset class="secupress-boxed-group">
			<legend class="screen-reader-text"><span><?php _e( 'DataBase Tables', 'secupress' ); ?></span></legend>

			<b><?php _e( 'Unknown tables', 'secupress' ); ?></b>
			<br>
			<?php
			if ( $other_tables ) {
				$chosen_tables = get_site_option( 'secupress_database-backups_settings' );

				if ( is_array( $chosen_tables ) && isset( $chosen_tables['other_tables'] ) && is_array( $chosen_tables['other_tables'] ) ) {
					$chosen_tables = array_intersect( $other_tables, $chosen_tables['other_tables'] );
					$chosen_tables = $chosen_tables ? array_flip( $chosen_tables ) : array();
				} else {
					$chosen_tables = array_flip( $other_tables );
				}

				// Skip our geoip table.
				if ( ! empty( $wpdb->prefix . 'secupress_geoips' ) ) {
					unset( $chosen_tables[ $wpdb->prefix . 'secupress_geoips' ] );
				}

				foreach ( $other_tables as $table ) {
					echo '<label><input name="other_tables[]" value="' . esc_attr( $table ) . '"' . ( isset( $chosen_tables[ $table ] ) ? ' checked="checked"' : '' ) . ' type="checkbox" class="secupress-checkbox secupress-checkbox-mini"> <span class="label-text">' . $table . '</span></label><br>';
				}
			} else {
				_ex( 'None', 'database table', 'secupress' );
			}
			?>
			<hr>
			<b><?php _e( 'WordPress tables (mandatory)', 'secupress' ); ?></b>
			<br>
			<?php
			if ( $wp_tables ) {
				foreach ( $wp_tables as $table ) {
					echo '<label><input disabled="disabled" checked="checked" type="checkbox" class="secupress-checkbox secupress-checkbox-mini"> <span class="label-text">' . $table . '</span></label><br>';
				}
			} else {
				_ex( 'None', 'database table', 'secupress' );
			}
			?>
		</fieldset>

		<p class="submit">
			<button class="secupress-button" type="submit" data-original-i18n="<?php esc_attr_e( 'Backup my Database', 'secupress' ); ?>" data-loading-i18n="<?php esc_attr_e( 'Backuping&hellip;', 'secupress' ); ?>" id="submit-backup-db">
				<span class="icon">
					<i class="secupress-icon-download" aria-hidden="true"></i>
				</span>
				<span class="text">
					<?php _e( 'Backup my Database', 'secupress' ); ?>
				</span>
			</button>
			<span class="spinner secupress-inline-spinner"></span>
		</p>

	</form>
	<?php
}


/**
 * Content of the settings field that displays the files to backup.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_backup_files_field( $args, $instance ) {
	$ignored_directories = get_site_option( 'secupress_file-backups_settings' );
	$ignored_directories = ! empty( $ignored_directories['ignored_directories'] ) ? $ignored_directories['ignored_directories'] : '';
	?>
	<form action="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_backup_files' ), 'secupress_backup_files' ) ); ?>" id="form-do-files-backup" method="post">

		<fieldset>
			<legend><strong><label for="ignored_directories"><?php _e( 'Do not backup the following files and folders:', 'secupress' ); ?></label></strong></legend>
			<br>
			<textarea id="ignored_directories" name="ignored_directories" cols="50" rows="5"><?php echo esc_textarea( $ignored_directories ); ?></textarea>
			<p class="description">
				<?php _e( 'One file or folder per line.', 'secupress' ); ?>
			</p>
		</fieldset>

		<p class="submit">
			<button class="secupress-button" type="submit" data-original-i18n="<?php esc_attr_e( 'Backup my files', 'secupress' ); ?>" data-loading-i18n="<?php esc_attr_e( 'Backuping&hellip;', 'secupress' ); ?>" id="submit-backup-files">
				<span class="icon">
					<i class="secupress-icon-download" aria-hidden="true"></i>
				</span>
				<span class="text">
					<?php _e( 'Backup my Files', 'secupress' ); ?>
				</span>
			</button>
			<span class="spinner secupress-inline-spinner"></span>
		</p>

	</form>
	<?php
}


/**
 * Content of the settings field for the scheduled backups.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_scheduled_backups_field( $args, $instance ) {
	$submodule     = 'schedules-backups';
	$next_schedule = secupress_get_next_scheduled_backup();

	if ( ! $next_schedule ) {
		_e( 'Nothing’s scheduled yet.', 'secupress' );
		return;
	}

	// Date.
	$next_schedule = date_i18n( _x( 'l, F jS, Y \a\t h:ia', 'Schedule date', 'secupress' ), $next_schedule );

	// What to backup.
	$def_types = array( 'db', 'files' );
	$types     = secupress_get_module_option( $submodule . '_type', array(), 'schedules' );
	$types     = is_array( $types ) ? $types : array();
	$types     = array_intersect( $def_types, $types );
	$types     = $types ? $types : $def_types;
	$types     = array_flip( $types );
	$type      = array();

	if ( isset( $types['db'] ) ) {
		$type[] = __( 'the database', 'secupress' );
	}
	if ( isset( $types['files'] ) ) {
		$type[] = __( 'the files', 'secupress' );
	}

	// Periodicity.
	$periodicity = (int) secupress_get_module_option( $submodule . '_periodicity', 1, 'schedules' );

	// Email.
	$email = secupress_get_module_option( $submodule . '_email', '', 'schedules' );
	$email = $email ? is_email( $email ) : false;

	printf(
		/** Translators: 1 is a date, 2 is "the database", "the files", or "the database and the files". */
		__( 'Next backup will occur %1$s and will backup %2$s.', 'secupress' ),
		'<span class="secupress-highlight">' . $next_schedule . '</span>',
		'<span class="secupress-highlight">' . wp_sprintf( '%l', $type ) . '</span>'
	);

	echo "<br/>\n";

	if ( $email ) {
		printf(
			/** Translators: 1 is "repeat every X days", 2 is an email address. */
			__( 'This task will %1$s, and once finished you will be notified at: %2$s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>',
			'<span class="secupress-highlight">' . $email . '</span>'
		);
	} else {
		printf(
			/** Translators: %s is "repeat every X days". */
			__( 'This task will %s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>'
		);
	}
}


/**
 * Content of the settings field for the scheduled scan.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_scheduled_scan_field( $args, $instance ) {
	$submodule     = 'schedules-scan';
	$next_schedule = secupress_get_next_scheduled_scan();

	if ( ! $next_schedule ) {
		_e( 'Nothing’s scheduled yet.', 'secupress' );
		return;
	}

	// Date.
	$next_schedule = date_i18n( _x( 'l, F jS, Y \a\t h:ia', 'Schedule date', 'secupress' ), $next_schedule );

	// Periodicity.
	$periodicity = (int) secupress_get_module_option( $submodule . '_periodicity', 1, 'schedules' );

	// Email.
	$email = secupress_get_module_option( $submodule . '_email', '', 'schedules' );
	$email = $email ? is_email( $email ) : false;

	printf(
		/** Translators: %s is a date. */
		__( 'Next scan will occur %s.', 'secupress' ),
		'<span class="secupress-highlight">' . $next_schedule . '</span>'
	);

	echo "<br/>\n";

	if ( $email ) {
		printf(
			/** Translators: 1 is "repeat every X days", 2 is an email address. */
			__( 'This task will %1$s, and once finished you will be notified at: %2$s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>',
			'<span class="secupress-highlight">' . $email . '</span>'
		);
	} else {
		printf(
			/** Translators: %s is "repeat every X days". */
			__( 'This task will %s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>'
		);
	}
}


/**
 * Content of the settings field for the scheduled file monitoring.
 *
 * @since 1.0
 *
 * @param (array)  $args     An array of parameters. See `SecuPress_Settings::field()`.
 * @param (object) $instance SecuPress_Settings object.
 */
function secupress_pro_scheduled_monitoring_field( $args, $instance ) {
	$submodule     = 'schedules-file-monitoring';
	$next_schedule = secupress_get_next_scheduled_file_monitoring();

	if ( ! $next_schedule ) {
		_e( 'Nothing’s scheduled yet.', 'secupress' );
		return;
	}

	// Date.
	$next_schedule = date_i18n( _x( 'l, F jS, Y \a\t h:ia', 'Schedule date', 'secupress' ), $next_schedule );

	// Periodicity.
	$periodicity = (int) secupress_get_module_option( $submodule . '_periodicity', 1, 'schedules' );

	// Email.
	$email = secupress_get_module_option( $submodule . '_email', '', 'schedules' );
	$email = $email ? is_email( $email ) : false;

	printf(
		/** Translators: %s is a date. */
		__( 'Next scan will occur %s.', 'secupress' ),
		'<span class="secupress-highlight">' . $next_schedule . '</span>'
	);

	echo "<br/>\n";

	if ( $email ) {
		printf(
			/** Translators: 1 is "repeat every X days", 2 is an email address. */
			__( 'This task will %1$s, and once finished you will be notified at: %2$s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>',
			'<span class="secupress-highlight">' . $email . '</span>'
		);
	} else {
		printf(
			/** Translators: %s is "repeat every X days". */
			__( 'This task will %s.', 'secupress' ),
			'<span class="secupress-highlight">' . sprintf( _n( 'repeat every %d day', 'repeat every %d days', $periodicity, 'secupress' ), $periodicity ) . '</span>'
		);
	}
}
