<?php
defined( 'ABSPATH' ) or die( 'Something went wrong.' );

add_action( 'init', 'secupress_init_get_malware_files' );
/**
 * Initiate the cron that get the malware files
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_init_get_malware_files() {
	if ( ! secupress_is_pro() ) {
		return;
	}
	if ( ! wp_next_scheduled( 'secupress_malware_files' ) ) {
		wp_schedule_event( time(), 'daily', 'secupress_malware_files' );
	}
}

add_action( 'secupress_malware_files', 'secupress_malware_files_cron' );
add_action( 'admin_post_secupress_update_malware_data', 'secupress_malware_files_cron' );
add_action( 'wp_ajax_secupress_update_malware_data', 'secupress_malware_files_cron' );
/**
 * Cron that will get the malware files.
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_malware_files_cron() {
	$doing_adminpost = 'admin_post_secupress_update_malware_data' === current_action();
	if ( $doing_adminpost ) {
		check_admin_referer( 'secupress_update_malware_data' );
		if ( ! current_user_can( secupress_get_capability() ) ) {
			wp_die( 'Something went wrong.' );
		}
	}
	if ( wp_doing_ajax() ) {
		check_ajax_referer( 'secupress_update_malware_data' );
		if ( ! current_user_can( secupress_get_capability() ) ) {
			die( '-1' );
		}
	}
	$api_type      = 'data';
	$local_version = secupress_get_option( 'malwaredb_version' );
	$api_url       = SECUPRESS_API_MAIN . $api_type . '/v2/'; // Only get the version in text, freely.
	$response      = wp_remote_get( $api_url );
	secupress_require_class( 'Admin', 'Notices' );
	if ( ! function_exists( 'secupress_add_transient_notice' ) ) {
		require_once SECUPRESS_INC_PATH . 'admin/functions/notices.php';
	}
	if ( ! is_wp_error( $response ) && ! empty( wp_remote_retrieve_body( $response ) ) ) {
		$filesystem      = secupress_get_filesystem();
		$json            = json_decode( wp_remote_retrieve_body( $response ), true );
		if ( json_last_error() > 0 ) {
			secupress_add_transient_notice( sprintf( __( 'The malware database files could not be updated due to a JSON issue: Error %1$s. Please %2$scontact us%3$s for more information', 'secupress' ), secupress_code_me( '#' . json_last_error() ), '<a href="' . SECUPRESS_WEB_MAIN . _x( 'support', 'link to website (Only FR or EN!)', 'secupress' ) . '">', '</a>' ), 'error' );
			if ( $doing_adminpost ) {
				wp_redirect( wp_get_referer() );
				die();
			} elseif ( wp_doing_ajax() ) {
				die( '-2' );
			}
			return; 
		}
		$remote_version  = $json['version'];
		$required_sp_ver = $json['required'];
		$types           = [ 'plugin' => [], 'theme' => [] ];

		unset( $response, $json );
		// Is the remote db version ok with this SP version here?
		if ( version_compare( SECUPRESS_VERSION, $required_sp_ver ) < 0 ) {
			secupress_add_transient_notice( sprintf( __( 'The malware database file could not be updated due to a %1$s version issue. Minimum version required: %2$s. Please update the plugin first.', 'secupress' ), SECUPRESS_PLUGIN_NAME, secupress_code_me( sanitize_key( $required_sp_ver ) ) ), 'error' );
			if ( $doing_adminpost ) {
				wp_redirect( wp_get_referer() );
				die();
			} elseif ( wp_doing_ajax() ) {
				die( '-3' );
			}
			return;
		}
		$malware_data_exists = secupress_get_data_file_path( 'malware_keywords' );
		// Do we have to update the malware DB?
		if ( secupress_is_pro() && 
			(
				( empty( $local_version ) || $local_version !== $remote_version ) || // No need to use version_compare(), if different, it's needed.
				! $malware_data_exists // Files does not exists? Rewrite them.
			) 
		) {
			$response            = wp_remote_post( $api_url, [ 'headers' => secupress_get_basic_auth_headers() ] );
			/**
			 * Change the download format.
			 * zip = 4MB, json = 40MB
			 *
			 * @author Julio Potier
			 * @since 2.2.6
			 * @param (string) $file_format
			 * @param (string) $api_type 'data' or 'geoip'
			 * 
			 * @return (string) 'zip' or 'json'
			 **/
			$file_format         = apply_filters( 'secupress.download_from_api.format', 'zip', $api_type );
			$downloaded_file     = secupress_download_from_api( $file_format, $api_type );

			if ( ! is_wp_error( $downloaded_file ) ) {
				$temp_dir        = dirname( $downloaded_file ) . '/';
				$options         = get_site_option( SECUPRESS_SETTINGS_SLUG );
				$xheader         = wp_remote_retrieve_header( $response, 'x-headers' );
				$header_license  = ! empty( $xheader ) ? trim( $xheader ) : null;

				// LICENSE FROM SP SAVED HERE
				if ( $header_license ) {
					$options['license'] = wp_parse_args( json_decode( $header_license, true ), [ 'limit' => 1, 'count' => 1, 'status' => 'active' ] ); // Basically, you have 1 and use 1 and active.
				}
				switch( $file_format ) {
					case 'zip':
						global $wp_filesystem;
						$wp_filesystem = secupress_get_filesystem();
						unzip_file( $downloaded_file, $temp_dir );
						unlink( $downloaded_file );
						foreach( secupress_get_data_file_paths() as $path => $keys ) {
							foreach( $keys as $key ) {
								if ( ! file_exists( $temp_dir . $key . '.data' ) ) {
									continue;
								}
								rename( $temp_dir . $key . '.data', $path . $key . '.data' );
							}
						}
						$wp_filesystem->put_contents( SECUPRESS_PRO_INC_PATH . 'auth.data', var_export( secupress_get_basic_auth_headers(), true ) );
					break;

					case 'json':
						if ( file_exists( $downloaded_file ) ) {
							$contents = file_get_contents( $downloaded_file );
							unlink( $downloaded_file );
							$contents = ! empty( $contents ) ? json_decode( $contents, true ) : [];
							if ( empty( $contents['data'] ) ) {
								secupress_add_transient_notice( sprintf( __( 'Empty content data from API with format %s.', 'secupress' ), secupress_code_me( esc_html( $file_format ) ) ), 'error', 'malware-update' );
								if ( $doing_adminpost ) {
									wp_redirect( wp_get_referer() );
									die();
								} elseif ( wp_doing_ajax() ) {
									die( '-4' );
								}
								return;
							}
							$contents = $contents['data'];
							foreach( secupress_get_data_file_paths() as $path => $keys ) {
								foreach( $keys as $key ) {
									$filesystem->put_contents( $path . $key . '.data', ( isset( $contents[ $key ] ) ? $contents[ $key ] : '[]' ) ); // Empty json
								}
							}
						}
						$wp_filesystem->put_contents( SECUPRESS_PRO_INC_PATH . 'auth.data', var_export( secupress_get_basic_auth_headers(), true ) );
					break;
					
					default:
						secupress_add_transient_notice( sprintf( __( 'Undefined file format when downloading from API: %s<br>Expected ZIP or JSON.', 'secupress' ), secupress_code_me( esc_html( $file_format ) ) ), 'error', 'malware-update' );
					break;
					
				}

				$options['malwaredb_version']     = $remote_version;
				$options['malwaredb_last_update'] = date_i18n( get_option( 'date_format' ), time() );
				secupress_update_options( $options );
				secupress_add_transient_notice( sprintf( __( 'The malware database file have been updated to version %s.', 'secupress' ), $remote_version, 'malware-update' ) );

				unset( $response, $contents );
				if ( wp_doing_ajax() ) {
					wp_send_json_success();
				}
			// } elseif ( 401 === wp_remote_retrieve_response_code( $response ) ) {
			// 	secupress_add_transient_notice( sprintf( __( 'The malware database file could not be downloaded due to a license issue. Please contact <a href="%s">the support</a>.', 'secupress' ), SECUPRESS_WEB_MAIN . _x( 'support', 'link to website (Only FR or EN!)', 'secupress' ) ), 'error' );
			// 	return; 
			} else {
				//// secupress_add_transient_notice( sprintf( __( 'An error occurred while downloading the malware database file. Please try again later.<br>Error %s', 'secupress' ), secupress_code_me( '#' . __LINE__ ) ), 'error' );
			}
		}
	} else {
		if ( ! is_null( $response ) ) {
			//secupress_add_transient_notice( sprintf( __( 'An error occurred while reading the malware database version. Please open a %ssupport ticket%s', 'secupress' ), '<a href="' . esc_url( SECUPRESS_WEB_MAIN . _x( 'support', 'link to website (Only FR or EN!)', 'secupress' ) ) . '">', '</a>' ), 'error' );
		}
	}
	if ( $doing_adminpost ) {
		wp_redirect( wp_get_referer() );
		die();
	} elseif ( wp_doing_ajax() ) {
		die( '-5' );
	}
	return; 
}

add_filter( 'cron_schedules', 'secupress_cron_add_weekly' );
/**
 * Initiate the new cron schedules
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (array) $schedules
 */
function secupress_cron_add_weekly( $schedules ) {
	if ( ! isset( $schedules['weekly'] ) ) {
		$schedules['weekly'] = [
			'interval'       => WEEK_IN_SECONDS,
			'display'        => __( 'Every week', 'secupress' )
		];
	}

	return $schedules;
}

add_action( 'init', 'secupress_init_license_check_cron' );
/**
 * Initiate the cron that will check for the license validity twice-daily.
 *
 * @since 2.2.6 monthly
 * @author Julio Potier
 *
 * @since 1.0.3
 * @author Grégory Viguier
 */
function secupress_init_license_check_cron() {
	if ( ! secupress_is_pro() ) {
		return;
	}
	if ( ! wp_next_scheduled( 'secupress_license_check' ) ) {
		wp_schedule_event( time(), 'monthly', 'secupress_license_check' );
	}
}

add_action( 'secupress_license_check', 'secupress_license_check_cron' );
/**
 * Cron that will check for the license validity.
 *
 * @since 1.0.3
 * @author Grégory Viguier
 */
function secupress_license_check_cron() {
	if ( ! secupress_is_pro() ) {
		return;
	}

	$headers  = secupress_get_basic_auth_headers( secupress_get_consumer_email(), secupress_get_consumer_key() );
	$url      = SECUPRESS_API_MAIN . 'key/v2/?sp_action=check_pro_license';
	$response = wp_remote_get( $url, [ 'timeout' => 10, 'headers' => $headers ] );

	if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
		return;
	}

	$body = wp_remote_retrieve_body( $response );
	$body = @json_decode( $body );

	if ( ! is_object( $body ) ) {
		return;
	}

	if ( ! empty( $body->success ) && ! empty( $body->data->site_is_pro ) ) {
		// The license is fine.
		return;
	}

	$options = get_site_option( SECUPRESS_SETTINGS_SLUG );
	$options = is_array( $options ) ? $options : array();
	unset( $options['site_is_pro'] );

	if ( ! empty( $body->data->error ) ) {
		// The error code returned by EDD.
		$options['license_error'] = esc_html( $body->data->error );
	}

	secupress_update_options( $options );
}

add_action( 'template_redirect', 'secupress_talk_to_me' );
/**
 * If plugin is active and license is + correct API (client) key is given, it will print the installed version
 *
 * @since 1.4.6
 * @author Julio Potier
 **/
function secupress_talk_to_me() {
	global $wp_version;
	if ( ! isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) || false === strpos( $_SERVER['HTTP_X_REQUESTED_WITH'], 'SECUPRESS' ) ) {
		return;
	}
	$consumer_key   = secupress_get_option( 'consumer_key', '' );
	$consumer_email = secupress_get_option( 'consumer_email', '' );
	$api_key        = str_replace( 'SECUPRESS_API_KEY:', '', $_SERVER['HTTP_X_REQUESTED_WITH'] );
	if ( hash_equals( $consumer_key, $api_key ) && ( ! empty( $consumer_key ) && ! empty( $consumer_email ) ) ) {
		$items          = array_filter( (array) get_site_option( SECUPRESS_SCAN_TIMES ) );
		$reports        = array();
		$last_report    = '—';
		$time_offset    = get_option( 'gmt_offset' ) * HOUR_IN_SECONDS;

		if ( $items ) {
			$last_report = end( $items );
			$last_report = date_i18n( _x( 'M dS, Y \a\t h:ia', 'Latest scans', 'secupress' ), $last_report['time'] + $time_offset );
		}
		wp_send_json_success( [ // Need the registered licence info to prevent nulled infos in our backend.
								'license_key'      => $consumer_key,
								'license_email'    => $consumer_email,
								'spversion'        => SECUPRESS_VERSION,
								'wpversion'        => $wp_version,
								'msversion'        => (int) is_multisite(),
								'phpversion'       => phpversion(),
								'scanner_counts'   => secupress_get_scanner_counts(),
								'scanners_results' => secupress_get_scan_results(),
								'last_report'      => $last_report,
								'attacks'          => secupress_get_attacks(),
								'move_login_slug'  => secupress_get_module_option( 'move-login_slug-login', 'login', 'users-login' ),
							] );
	} elseif ( ! $consumer_key ) {
		wp_send_json_error( 'license_key' );
	}
}
