<?php
defined( 'ABSPATH' ) or die( 'Something went wrong.' );


/**
 * Get available alert types.
 *
 * @since 1.0
 *
 * @return (array) Return an array with identifiers as keys and field labels as values.
 */
function secupress_alert_types_labels() {

	return array(
		/* Active now */
			// 1.0
		'email'   => __( 'Email', 'secupress' ),
			// 2.0
		'slack'   => 'Slack',
		/* Not active yet */
		// 'twitter' => 'Twitter',
		// 'sms'     => 'SMS',
		// 'push'    => __( 'Push notification', 'secupress' ),
	);
}


/**
 * Send a notification validation
 *
 * @since 2.0
 * @author Julio Potier
 *
 * @see https://app.slack.com/block-kit-builder/
 *
 * @param (string) $type emails, sms, push, slack or custom, see secupress.notification.type
 * @param (array) $args Custom args
 * @return
 **/
function secupress_send_notification_validation( $type, $args = false ) {
	$current_user = wp_get_current_user();

	switch( $type ) {
		case 'emails':
			//
		break;
		case 'sms':
			//
		break;
		case 'push':
			//
		break;
		case 'slack':
			// Do not use secupress_maybe_reset_slack_notifs() here because the URL is not valid yet, this is the validation call.
			$url      = isset( $args['url'] ) ? $args['url'] : false;
			if ( ! $url ) {
				secupress_set_option( 'notification-types_slack', 0 );
				break;
			}
			$accepted = secupress_get_option( 'notification-types_slack', false );
			if ( $accepted || $url === $accepted ) {
				break;
			}
			secupress_set_option( 'notification-types_slack', 0 );

			$payload  = (object) array(
			   'attachments' =>
			  array (
			    0 =>
			    (object) array(
			       'color' => '#2BCDC1',
			       'blocks' =>
			      array (
			        0 =>
			        (object) array(
			           'type' => 'header',
			           'text' =>
			          (object) array(
			             'type' => 'plain_text',
			             'text' => sprintf( __( '%s Notifications', 'secupress' ), SECUPRESS_PLUGIN_NAME ),
			             'emoji' => true,
			          ),
			        ),
			        1 =>
			        (object) array(
			           'type' => 'section',
			           'text' =>
			          (object) array(
			             'type' => 'mrkdwn',
			             'text' => sprintf( __( '*%1$s* from %2$s requested to send the _SecuPress Notifications_ on this channel.', 'secupress' ), $current_user->display_name, home_url() )
			          ),
			           'block_id' => 'textInfo',
			        ),
			        2 =>
			        (object) array(
			           'type' => 'section',
			           'text' =>
			          (object) array(
			             'type' => 'mrkdwn',
			             'text' => "—\n" . __( 'Click *Accept*, or *ignore* this message.', 'secupress' ),
			          ),
			           'accessory' =>
			          (object) array(
			             'type' => 'image',
			             'image_url' => 'https://pbs.twimg.com/profile_images/737217340950077440/2Q06P22n_400x400.jpg',
			             'alt_text' => 'SecuPress Logo',
			          ),
			        ),
			        3 =>
			        (object) array(
			           'type' => 'section',
			           'text' =>
			          (object) array(
			             'type' => 'mrkdwn',
			             'text' => sprintf( __( 'Only *%s* can accept this request.', 'secupress' ), $current_user->display_name ),
			          ),
			           'accessory' =>
			          (object) array(
			             'type' => 'button',
						 'style' => 'primary',
			             'text' =>
			            (object) array(
			               'type' => 'plain_text',
			               'text' => '✅ ' . _x( 'Accept', 'verb', 'secupress' ),
			               'emoji' => true,
			            ),
			             'value' => 'acceptNotif',
			             'url' => str_replace( '&amp;', '&', wp_nonce_url( admin_url( 'admin-post.php?action=secupress_accept_notification&type=' . $type ), 'secupress_accept_notification-type-' . $type ) ),
			             'action_id' => 'button-action',
			          ),
		            ),
			      ),
			    ),
			  ),
			);
			secupress_send_slack_notification( $payload, true ); // true to bypass acceptation because it's not already accepted but we need to!

		break;
		case 'twitter':
			//
		break;
		default:
			/**
			* Manage the possible new filtered notification type
			* @see 'secupress.notifications.types'
			* @since 2.0
			* @author Julio Potier
			*/
			do_action( 'secupress.notification.type.' . $type );
		break;
	}
	/**
	* Manage the possible new filtered notification type
	* @since 2.0
	* @author Julio Potier
	*
	* @param (string) $type
	*/
	do_action( 'secupress.notification.type', $type );
}

/**
 * Send a Slack notification if the URL is correct
 *
 * @since 2.0
 * @author Julio Potier
 *
 * @param (string|array) $message Can be a Slack Bloc Kit or a simple string
 * @param (bool|string) $bypass True or url will bypass the accepted flag
 **/
function secupress_send_slack_notification( $message, $bypass = false ) {
	/**
	* Can force notifications to be sent without acceptation
	*
	* @author Julio Potier
	* @since 2.0
	*/
	$bypass = apply_filters( 'secupress.notifications.slack.bypass', $bypass );
	$url    = secupress_get_module_option( 'notification-types_slack', false, 'alerts' );
	if ( ! $url && ! $bypass ) {
		secupress_set_option( 'notification-types_slack', 0 );
		return;
	}
	$accepted = secupress_get_option( 'notification-types_slack', false );
	if ( ! $bypass && ( ! $accepted || $url !== $accepted ) ) {
		return;
	}
	if ( ! $url && is_bool( $bypass ) ) {
		return;
	}
	if ( ! $url && true === filter_var( $bypass, FILTER_VALIDATE_URL) ) {
		$url = $bypass;
	}
	$message = is_string( $message ) ? [ 'text' => $message ] : $message;
	$args    = [
				'headers' =>
				[
					'Content-type' => 'application/json',
					'blocking'     => false,
					'timeout'      => 0.01
				],
				'body' => json_encode( $message )
			];

	wp_remote_post( $url, $args );
}

/**
 * Return the modules alerts set in DB
 *
 * @since 2.3.18
 * @author Julio Potier
 * 
 * @return (array)
 **/
function secupress_get_submodule_alerts() {
	return get_option( SECUPRESS_MODULE_ALERTS, [] );
}

/**
 * Add a module alert in DB
 *
 * @since 2.3.18
 * @author Julio Potier
 * 
 * @param (string) $module
 * @param (string) $submodule
 * 
 * @return (bool) False if feature not active
 **/
function secupress_set_submodule_alert( $module, $submodule ) {
	if ( ! secupress_is_submodule_active( 'alerts', 'module-alerts' ) ) {
		return false;
	}
	$alerts                           = secupress_get_submodule_alerts();
	$alerts["{$module}-{$submodule}"] = [ 
		'module'      => $module, 
		'submodule'   => $submodule, 
		'time'        => time(), 
		'module_name' => secupress_get_module_data( $module, $submodule )['Name'], 
		'user_id'     => get_current_user_id()
	];
	return update_option( SECUPRESS_MODULE_ALERTS, $alerts );
}

/**
 * Remove a module alert from DB
 *
 * @since 2.3.18
 * @author Julio Potier
 * 
 * @param (string) $module
 * @param (string) $submodule
 **/
function secupress_remove_submodule_alert( $module, $submodule ) {
	$alerts = secupress_get_submodule_alerts();
	unset( $alerts["{$module}-{$submodule}"] );
	update_option( SECUPRESS_MODULE_ALERTS, $alerts );
}

/**
 * Send emails if some module alerts are found
 *
 * @since 2.3.18
 * @author Julio Potier
 * 
 * @param (array) $args
 **/
function secupress_submodule_alert_email( $args ) {
	$main_user     = get_user_by( 'ID', $args['user_id'] );
	if ( ! is_a( $main_user, 'WP_User' ) ) {
		return;
	}
	$module_name     = $args['module_name'];
	$timingf         = secupress_get_submodule_alert_timing( $args['time'] );
	$other_users     = secupress_alerts_get_emails();
	$email_subject   = apply_filters( 'secupress.plugins.module_alert.email_subject', sprintf( __( '[%s] Deactivated Module "%s"!', 'secupress' ), '###SITENAME###', esc_html( $module_name ) ), $args );
	$body_raw        = __(
		'Hi ###USERNAME###,

Reminder: %1$s deactivated the module "%2$s" on ###SITENAME### %3$s.

Please reactivate it if needed, or simply ignore this notice since no action is required.

This email has been sent to every other administrator on this site.

Regards,
All at ###SITENAME###
###SITEURL###', 'secupress' );
	$body_main       = apply_filters( 'secupress.plugins.module_alert.email_body.main', $body_raw, $args );
	$body_main       = sprintf( $body_main, _x( 'you', 'subject', 'secupress' ), $module_name, $timingf );
	$body_main       = str_replace( '###USERNAME###', $main_user->display_name, $body_main );
	secupress_send_mail( $main_user->user_email, $email_subject, $body_main );
	
	if ( $other_users ) {
		$body_others = apply_filters( 'secupress.plugins.module_alert.email_body.others', $body_raw, $args );
		$body_others = sprintf( $body_others, esc_html( $main_user->display_name ), $module_name, $timingf );
		foreach ( $other_users as $other_email ) {
			$_body_tmp = $body_others; // backup to prevent overwrite
			$other_user  = get_user_by( 'email', $other_email );
			// This is a user from this site
			if ( isset( $other_user->user_email ) ) {
				$_body_tmp = str_replace( '###USERNAME###', $other_user->display_name, $_body_tmp );
				secupress_send_mail( $other_user->user_email, $email_subject, $_body_tmp );
			} else {
				// This is the admin email!
				if ( $other_email === get_option( 'admin_email' ) ) {
					$_body_tmp = str_replace( '###USERNAME###', secupress_translate_user_role( 'administrator' ), $body_others );
				}
				$_body_tmp = str_replace( '###USERNAME###', '', $_body_tmp );
				secupress_send_mail( $other_email, $email_subject, $_body_tmp );
			}
		}
	}
}