<?php
defined( 'ABSPATH' ) or die( 'Something went wrong.' );


/**
 * File Monitoring class.
 *
 * @package SecuPress
 * @since 1.0
 */
class SecuPress_File_Monitoring extends SecuPress_Singleton {

	const VERSION = '1.0';

	/**
	 * Singleton The reference to *Singleton* instance of this class.
	 *
	 * @var (object)
	 */
	protected static $_instance;

	/**
	 * SecuPress_Background_Process_File_Monitoring instance.
	 *
	 * @var (object)
	 */
	protected $background_process;


	/** Public methods ========================================================================== */

	/**
	 * Add tasks to the queue and dispatch.
	 *
	 * @since 1.0
	 *
	 * @return (bool) True on success. False if it was already running.
	 */
	public function do_file_scan() {
		global $wp_version, $wp_local_package;

		if ( $this->is_monitoring_running() ) {
			return false;
		}

		// Cleanup previous batches.
		$this->stop_file_scan();

		$wp_core_files_hashes = get_site_option( SECUPRESS_WP_CORE_FILES_HASHES );

		if ( false === $wp_core_files_hashes || empty( $wp_core_files_hashes[ $wp_version ] ) ) {
			if ( ! $this->background_process->push_to_queue( 'get_wp_hashes' ) ) {
				return false;
			}
		}

		if ( isset( $wp_local_package, $wp_core_files_hashes['locale'] ) && $wp_core_files_hashes['locale'] !== $wp_local_package ) {
			$fix_dists = get_site_option( SECUPRESS_FIX_DISTS );

			if ( false === $fix_dists || ! isset( $fix_dists[ $wp_version ] ) ) {
				$this->background_process->push_to_queue( 'fix_dists' );
			}
		}

		$this->background_process->push_to_queue( 'scan_full_tree' )->save()->dispatch();
		update_option( SECUPRESS_MALWARE_SCAN_LAST_TIME, time() );
		return true;
	}


	/**
	 * Launch DB scan
	 *
	 * @since 2.0
	 */
	public function do_database_scan() {
		global $wpdb;
		$keywords = secupress_get_database_malware_keywords();

		if ( empty( $keywords ) ) {
			return [];
		}

		$reqs             = [];
		$sqls             = [];
		$sqls['posts']    = 'SELECT ID, post_content from ' . $wpdb->posts . ' WHERE ( ';
		$sqls['opt_val']  = 'SELECT option_name, option_value from ' . $wpdb->options . ' WHERE ( ';
		// $sqls['cpts']     = 'SELECT ID, post_content from ' . $wpdb->posts . ' WHERE ( 1=0 ';
		// $sqls['opts']     = 'SELECT option_name, option_value from ' . $wpdb->options . ' WHERE ( 1=0 ';
		// Build SQL queries first.
		foreach ( $keywords as $key => $all ) {
			$or           = '';
			switch( $key ) {
				case 'posts':
					foreach( $all as $items ) {
						$sqls['opt_val'] .= "$or( 1=1 ";
						$sqls['posts']   .= "$or( 1=1 ";
						foreach( $items as $item ) {
							$sqls['posts']   .= 'AND post_content LIKE "%' . ( $item ) . '%"';
							$sqls['opt_val'] .= 'AND option_value LIKE "%' . ( $item ) . '%"';
							$or               = ' OR ';
						}
						$sqls['posts']   .= ')';
						$sqls['opt_val'] .= ')';
					}
				break;
				// case 'cpts':
					// 	foreach( $all as $item ) {
						// 		$sqls[ $key ] .= ' OR post_type = "' . ( $item ) . '"';
						// 	}
				// break;
				// case 'opts':
					// 	foreach( $all as $item ) {
						// 		$sqls[ $key ] .= ' OR option_name = "' . ( $item ) . '"';
						// 	}
				// break;
			}
		}
		$sqls['posts']   .= ')';
		$sqls['opt_val'] .= ')';
		// Run them.
		foreach ( $sqls as $key => $sql ) {
			$reqs[ $key ] = $wpdb->get_results( $sql, ARRAY_A );
		}
		// Save them.
		update_site_option( SECUPRESS_DATABASE_MALWARES, str_rot13( json_encode( array_filter( $reqs ) ) ) );
		return true;
	}



	/**
	 * Launch content spam scan
	 *
	 * @since 2.2.6
	 * 
	 * @return (bool) True to continue the scan. False to stop it.
	 */
	public function do_content_spam_scan() {
		if ( ! secupress_wp_version_is( '6.5' ) ) {
			// WP_HTML_Tag_Processor as we use it requires WP 6.5 or +.
			return true;
		}
		$allowed_list      = get_site_option( SECUPRESS_CONTENT_ALLOWED, [] );
		if ( ! empty( $allowed_list ) ) {
			$allowed_list  = json_decode( str_rot13( $allowed_list ), true );
		}
		if ( ! is_array( $allowed_list ) ) {
			$allowed_list  = [];
		}
		$allowed_list      = array_flip( $allowed_list );
		
		$contents          = [];
		$filename          = secupress_get_data_file_path( 'tag_attr' );
		$tags              = [];
		if ( file_exists( $filename ) ) {
			$tags          = explode( ',', file_get_contents( $filename ) );
		}
		/**
		 * Manage the tags to be filtered, format : "tag|attr", example "a|href"
		 * 
		 * @param (array) $tags
		 * @since 2.2.6
		 * @return (array) $tags
		 */
		$tags              = apply_filters( 'secupress.content-spam.scanner.tags', $tags );
		// So, nothing to scan?
		if ( empty( $tags ) ) {
			return false;
		}
		$tags              = array_fill_keys( $tags, [] );
		
		$urls              = [ home_url(), add_query_arg( 's', 'secupress-search-test-content-spam-' . time(), home_url() ), home_url( '/secupress-404-page-test-content-spam-' . time() . '.html' ) ];
		/**
		 * Manage the urls to be tested online to find external resources loaded from the website.
		 * 
		 * @param (array) $urls
		 * @since 2.2.6
		 * @return (array) $urls
		 */
		$urls              = apply_filters( 'secupress.content-spam.scanner.test_urls', $urls );

		foreach( $urls as $url ) {
			$response = wp_remote_get( $url );
			if ( is_wp_error( $response ) ) {
				continue;
			}
			$body = wp_remote_retrieve_body( $response );
			if ( ! empty( $body ) ) {
				$contents[ $url ] = $body;
			}
		}
		// So, nothing to scan?
		if ( empty( $contents ) ) {
			return false;
		}
		$allowed_domains   = [ trim( str_replace( [ 'https://', 'http://' ], '', home_url( '/' ) ), '/' ), 'w.org', 'wordpress.org', 'wp.org' ];
		/**
		 * Manage the allowed domains that does not need to be filtered.
		 * 
		 * @param (array) $allowed_domains
		 * @since 2.2.6
		 * @return (array) $allow_domains
		 */
		$allowed_domains   = apply_filters( 'secupress.content-spam.scanner.allowed_domains', $allowed_domains ); 

		foreach( $contents as $ID => $content ) {
			foreach( array_keys( $tags ) as $tag ) {
				$html = new WP_HTML_Tag_Processor( $content );
				list( $_tag, $_attr ) = explode( '|', $tag );
				while( $html->next_tag( $_tag ) ) {
					$attr    = $html->get_attribute( $_attr );
					if ( ! $attr ) {
						continue;
					}
					$attr    = explode( '#', $attr );
					$attr    = reset( $attr );
					if ( ! $attr ) {
						continue;
					}
					$attr    = strtolower( $attr );
					if ( strpos( $attr, 'http://' ) === false && strpos( $attr, 'https://' ) === false ) {
						continue;
					}
					$href_dom = parse_url( $attr, PHP_URL_HOST );
					if ( ! $href_dom ) {
						continue;
					}
					$href_dom = strtolower( $href_dom );
					
					$cache_key = $tag . '|' . $attr;
					if ( secupress_cache_data( $cache_key ) ) {
						continue;
					}
					
					$is_allowed = false;
					if ( ! empty( $allowed_domains ) ) {
						foreach( $allowed_domains as $allowed ) {
							$allowed = strtolower( trim( $allowed ) );
							if ( $href_dom === $allowed || strpos( $href_dom, 'www.' . $allowed ) === 0 ) {
								$is_allowed = true;
								break;
							}
						}
					}
					
					if ( ! $is_allowed && ! isset( $allowed_list[ $attr ] ) ) {
						if ( ! isset( $tags[ $tag ][ $attr ] ) ) {
							$tags[ $tag ][ $attr ] = $ID;
						}
						secupress_cache_data( $cache_key, 1 );
					}
				}
			}
		}
		// Save them.
		update_site_option( SECUPRESS_CONTENT_SPAM_SCAN, str_rot13( json_encode( $tags ) ) );

		return true;
	}



	/**
	 * Remove everything from the queue.
	 *
	 * @since 1.0
	 */
	public function stop_file_scan() {
		$this->background_process->delete_queue();
	}


	public function get_malware_scan_last_time() {
		return (int) get_option( SECUPRESS_MALWARE_SCAN_LAST_TIME, 0 );
	}


	public function delete_malware_scan_last_time() {
		delete_option( SECUPRESS_MALWARE_SCAN_LAST_TIME );
	}


	/**
	 * Is process running.
	 * Check whether the current process is already running in a background process.
	 *
	 * @since 1.0
	 *
	 * @return (bool)
	 */
	public function is_monitoring_running() {
		return $this->background_process->is_monitoring_running();
	}

	public function get_batch_id() {
		return $this->background_process->get_batch();
	}
	/** Private methods ========================================================================= */

	/**
	 * Class init.
	 *
	 * @since 1.0
	 */
	protected function _init() {
		secupress_require_class_async();

		require_once( SECUPRESS_PRO_MODULES_PATH . 'file-system/plugins/inc/php/file-monitoring/class-secupress-background-process-file-monitoring.php' );

		$this->background_process = new SecuPress_Background_Process_File_Monitoring;
	}
}
