<?php
/**
 * Module Name: Detect Bad Plugins
 * Description: Detect if a plugin you're using is known as vulnerable, closed, or removed from w.org.
 * Main Module: plugins_themes
 * Author: SecuPress
 * Version: 2.2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

add_action( 'secupress.plugins.loaded', 'secupress_bad_plugins_async_init' );
/**
 * Instantiate bad plugins class.
 *
 * @author Julio Potier
 * @since 2.2.6
 */
function secupress_bad_plugins_async_init() {
	add_filter( 'secupress_bad_plugins_cron_interval', function(){return 1;} );
	add_filter( 'secupress_bad_plugins_default_time_limit', function(){return 5;} );

	secupress_require_class_async();

	require_once( SECUPRESS_PRO_MODULES_PATH . 'plugins-themes/plugins/inc/php/class-secupress-background-process-bad-plugins.php' );

	SecuPress_Background_Process_Bad_Plugins::get_instance();

}

add_action( 'secupress_bad_plugins_maybe_do_checks', 'secupress_bad_plugins_maybe_do_checks' );
/**
 * Maybe process scheduled tests.
 *
 * @author Julio Potier
 * @since 2.2.6
 */
function secupress_bad_plugins_maybe_do_checks() {
	// Ensure classes are loaded before using them (important for wp-cron context)
	secupress_require_class_async();
	
	if ( ! class_exists( 'SecuPress_Background_Process_Bad_Plugins' ) ) {
		require_once( SECUPRESS_PRO_MODULES_PATH . 'plugins-themes/plugins/inc/php/class-secupress-background-process-bad-plugins.php' );
	}
	
	$process = SecuPress_Background_Process_Bad_Plugins::get_instance();
	if ( $process->is_processing() ) {
		return;
	}

	if ( ! function_exists ( 'get_plugins' ) ) {
		require_once( ABSPATH . '/wp-admin/includes/plugin.php' );
	}
	$retests  = [];
	$plugins  = array_keys( get_plugins() );
	foreach( $plugins as $plugin_file ) {
		$plugin_data = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file, false, false );
		if ( empty( $plugin_data['UpdateURI'] ) || strpos( $plugin_data['UpdateURI'], 'wordpress.org' ) !== false ) {
			$retests[] = $plugin_file;
		}
	}
	// Do retests asynchroniously.
	array_map( [ $process, 'push_to_queue' ], $retests );

	$process->save()->dispatch();
}

add_action( 'after_plugin_row', 'secupress_detect_bad_plugins_after_plugin_row', 10, 3 );
/**
 * Add a red banner on each "bad" plugin on plugins page
 *
 * @since 2.2.6 Better details
 * @author Julio Potier
 * @since 1.0
 * @author Grégory Viguier
 *
 * @param (string) $plugin_file Path to the plugin file.
 * @param (array)  $plugin_data Plufin data.
 * @param (string) $context     Context.
 */
function secupress_detect_bad_plugins_after_plugin_row( $plugin_file, $plugin_data, $context ) {
	static $plugins;

	if ( ( is_network_admin() || ! is_multisite() ) && ! current_user_can( 'update_plugins' ) && ! current_user_can( 'delete_plugins' ) && ! current_user_can( 'activate_plugins' ) ) { // Ie. Administrator.
		return;
	}
	if ( ! isset( $plugins ) ) {
		$plugins = [
			'vulns'  => secupress_get_bad_plugins( 'vulns' ),
			'closed' => secupress_get_bad_plugins( 'closed' ),
			'old'    => secupress_get_bad_plugins( 'old' ),
		];
	}
	$folder_for_is = dirname( $plugin_file );
	$is_closed     = isset( $plugins['closed'][ $plugin_file ] );
	$is_old        = isset( $plugins['old'][ $plugin_file ] );
	$is_vuln       = isset( $plugins['vulns'][ $folder_for_is ] );
	$plugin_vuln   = $is_vuln ? $plugins['vulns'][ $folder_for_is ] : false;
		if ( ! $is_closed && ! $is_vuln && ! $is_old ) {
		return;
	}
	if ( $is_vuln && $plugin_vuln['fixed_in'] && version_compare( $plugin_data['Version'], $plugin_vuln['fixed_in'] ) >= 0 ) {
		return;
	}

	$wp_list_table = _get_list_table( 'WP_Plugins_List_Table' );
	$current       = get_site_transient( 'update_plugins' );
	$page          = get_query_var( 'paged' );
	$s             = isset( $_REQUEST['s'] ) ? esc_attr( stripslashes( $_REQUEST['s'] ) ) : '';
	$r             = isset( $current->response[ $plugin_file ] ) ? $current->response[ $plugin_file ] : null;
	// HTML output.
	?>
	<tr class="secupress-bad-plugins">
		<td colspan="<?php echo $wp_list_table->get_column_count(); ?>">
			<div class="error-message notice inline notice-error notice-alt">
			<?php
			printf( '<em>' . sprintf( __( '%s Warning:', 'secupress' ), SECUPRESS_PLUGIN_NAME ) . '</em> ' );
			if ( $is_vuln ) {
				printf( _x( '<strong>%1$s %2$s</strong> is known to contain this vulnerability: %3$s.', 'unknow theme', 'secupress' ),
					$plugin_data['Name'],
					$plugin_vuln['fixed_in'] ? sprintf( __( 'version %s (or lower)', 'secupress' ), $plugin_vuln['fixed_in'] ) : __( 'all versions', 'secupress' ),
					'<strong>' . esc_html( $plugin_vuln['flaws'] ) . '</strong>'
				);

				echo ' <a href="' . esc_url( $plugin_vuln['refs'] ) . '" target="_blank">' . __( 'More information', 'secupress' ) . '</a>';

				if ( ! empty( $plugin_vuln['fixed_in'] ) && current_user_can( 'update_plugins' ) ) {
					echo '<p>';

					if ( ! empty( $r->package ) ) {
						echo '<span class="dashicons dashicons-update" aria-hidden="true"></span> ';
						printf(
							__( '%1$s invites you to <a href="%2$s">Update</a> this plugin to version %3$s.', 'secupress' ),
							SECUPRESS_PLUGIN_NAME,
							esc_url( wp_nonce_url( admin_url( 'update.php?action=upgrade-plugin&plugin=' ) . $plugin_file, 'upgrade-plugin_' . $plugin_file ) ),
							'<strong>' . esc_html( isset( $r->new_version ) ? $r->new_version : $plugin_vuln['fixed_in'] ) . '</strong>'
						);
					} elseif ( null !== $r ) {
						echo '<span class="dashicons dashicons-update" aria-hidden="true"></span> ' . sprintf( __( '%s invites you to Update this plugin <em>(automatic update is unavailable for this plugin.)</em>.', 'secupress' ), SECUPRESS_PLUGIN_NAME );
					} else {
						echo '<p><span class="dashicons dashicons-update" aria-hidden="true"></span> ' . __( 'Update is unavailable for this plugin.', 'secupress' ) . '</p>';
					}

					echo '</p>';
				} else {
					echo '<p>';

					if ( secupress_is_plugin_active( $plugin_file ) && current_user_can( 'activate_plugins' ) ) {
						printf(
							'<span class="dashicons dashicons-admin-plugins" aria-hidden="true"></span> ' . __( '%s invites you to <a href="%s">deactivate</a> this plugin, then delete it.', 'secupress' ),
							SECUPRESS_PLUGIN_NAME,
							esc_url( wp_nonce_url( admin_url( 'plugins.php?action=deactivate&plugin=' . $plugin_file . '&plugin_status=' . $context . '&paged=' . $page . '&s=' . $s ), 'deactivate-plugin_' . $plugin_file ) )
						);
					}

					if ( ! secupress_is_plugin_active( $plugin_file ) && current_user_can( 'delete_plugins' ) ) {
						if ( $is_closed ) {
							printf(
								'<span class="dashicons dashicons-trash" aria-hidden="true"></span> ' . __( '%s invites you to <a href="%s">delete</a> this plugin, as it has been removed from the official repository.', 'secupress' ),
								SECUPRESS_PLUGIN_NAME,
								esc_url( wp_nonce_url( admin_url( 'plugins.php?action=delete-selected&amp;checked[]=' . $plugin_file . '&amp;plugin_status=' . $context . '&amp;paged=' . $page . '&amp;s=' . $s ), 'bulk-plugins' ) )
							);
						} else {
						printf(
							'<span class="dashicons dashicons-trash" aria-hidden="true"></span> ' . __( '%s invites you to <a href="%s">delete</a> this plugin, as no patch has been made by its author.', 'secupress' ),
								SECUPRESS_PLUGIN_NAME,
								esc_url( wp_nonce_url( admin_url( 'plugins.php?action=delete-selected&amp;checked[]=' . $plugin_file . '&amp;plugin_status=' . $context . '&amp;paged=' . $page . '&amp;s=' . $s ), 'bulk-plugins' ) )
							);
						}
					}

					echo '</p>';
				}
			} elseif ( $is_closed ) {
				list( $date, $reason ) = explode( '|', $plugins['closed'][ $plugin_file ] );
				printf( _x( '<strong>%s</strong> has been removed from official repository since %s. Reason: %s', 'closed theme', 'secupress' ), esc_html( $plugin_data['Name'] ), esc_html( date_i18n( get_option( 'date_format' ), $date ) ), secupress_closed_reason_i18n( $reason, 'plugin' ) );
			} else { // Old.
				$y = secupress_minmax_range( 2, 20, (int) (new DateTime( '@'.time() ))->diff( new DateTime( '@'.$plugins['old'][ $plugin_file ] ) )->y );
				printf( _x( '<strong>%s</strong> has not been updated on official repository for more than %d years now (since %s). It can be dangerous to continue to use it.', 'updated theme', 'secupress' ), esc_html( $plugin_data['Name'] ), $y, esc_html( date_i18n( get_option( 'date_format' ), $plugins['old'][ $plugin_file ] ) ) );
			}

			if ( ! $is_vuln ) {
				echo '<p>';

				if ( secupress_is_plugin_active( $plugin_file ) && current_user_can( 'activate_plugins' ) ) {
					printf(
						'<span class="dashicons dashicons-admin-plugins" aria-hidden="true"></span> ' . __( '%s invites you to <a href="%s">deactivate</a> this plugin, then delete it.', 'secupress' ),
						SECUPRESS_PLUGIN_NAME,
						esc_url( wp_nonce_url( admin_url( 'plugins.php?action=deactivate&plugin=' . $plugin_file . '&plugin_status=' . $context . '&paged=' . $page . '&s=' . $s ), 'deactivate-plugin_' . $plugin_file ) )
					);
				}

				if ( ! secupress_is_plugin_active( $plugin_file ) && current_user_can( 'delete_plugins' ) ) {
					printf(
						'<span class="dashicons dashicons-trash" aria-hidden="true"></span> ' . __( '%s invites you to <a href="%s">delete</a> this plugin, as no patch has been made by its author.', 'secupress' ),
						SECUPRESS_PLUGIN_NAME,
						esc_url( wp_nonce_url( admin_url( 'plugins.php?action=delete-selected&amp;checked[]=' . $plugin_file . '&amp;plugin_status=' . $context . '&amp;paged=' . $page . '&amp;s=' . $s ), 'bulk-plugins' ) )
					);
				}

				echo '</p>';
			}
			?>
			</div>
		</td>
	</tr>
	<?php
}


add_action( 'admin_head', 'secupress_detect_bad_plugins_add_notices' );
/**
 * Add a notice if a plugin is considered as "bad"
 *
 * @since 1.0
 */
function secupress_detect_bad_plugins_add_notices() {
	global $pagenow;

	if ( 'plugins.php' === $pagenow || ( ( is_network_admin() || ! is_multisite() ) && ! current_user_can( secupress_get_capability() ) ) ) {
		return;
	}

	$plugins = array(
		'vulns'  => secupress_get_bad_plugins( 'vulns' ),
		'closed' => secupress_get_bad_plugins( 'closed' ),
		'old'    => secupress_get_bad_plugins( 'old' ),
	);

	if ( $plugins['vulns'] || $plugins['closed'] || $plugins['old'] ) {
		$counter = count( $plugins['vulns'] ) + count( $plugins['closed'] ) + count( $plugins['old'] );
		if ( ! $counter ) {
			return;
		}
		$url     = esc_url( admin_url( 'plugins.php' ) );
		$message = sprintf(
			_n(
				'Your installation contains %1$s plugin considered as <em>bad</em>, check the details in <a href="%2$s">the plugins page</a>.',
				'Your installation contains %1$s plugins considered as <em>bad</em>, check the details in <a href="%2$s">the plugins page</a>.',
				$counter,
				'secupress-pro'
			),
			secupress_tag_me( $counter, 'strong' ),
			$url
		);
		secupress_add_notice( $message, 'error', 'bad-plugins' );
	}
}

add_action( 'secupress.pro.plugins.activation',                                     'secupress_bad_plugins_activation' );
add_action( 'secupress.modules.activate_submodule_' . basename( __FILE__, '.php' ), 'secupress_bad_plugins_activation' );
/**
 * Initiate the cron that will check for vulnerable plugins.
 *
 * @since 2.2.6 Daily event now
 * @since 2.1
 * @author Julio Potier
 */
function secupress_bad_plugins_activation() {
	if ( ! wp_next_scheduled( 'secupress_bad_plugins' ) ) {
		wp_schedule_event( time(), 'daily', 'secupress_bad_plugins' );
		wp_schedule_event( time(), 'daily', 'secupress_bad_plugins_maybe_do_checks' );
	}
}

add_action( 'secupress_bad_plugins', 'secupress_detect_bad_plugins_async_get_and_store_infos' );
add_action( 'admin_post_secupress_bad_plugins_update_data', 'secupress_detect_bad_plugins_async_get_and_store_infos' );
/**
 * Save the data to refresh the vulnerable plugins. 
 * Moved from Pro to Free + renamed. Originally `secupress_detect_bad_plugins_async_get_infos()` (deprecated).
 * 
 *
 * @since 2.2.6 Change the URL API call
 * @since 2.1 Moved from /core/admin/admin.php, old hook "admin_footer" via admin-post using AJAX
 * @since 1.1.3
 */
function secupress_detect_bad_plugins_async_get_and_store_infos() {
	if ( ! wp_doing_cron() ) {
		check_admin_referer( 'secupress_bad_plugins_update_data' );
	}
	if ( ! function_exists ( 'get_plugins' ) ) {
		require_once( ABSPATH . '/wp-admin/includes/plugin.php' );
	}
	$plugins  = get_plugins();
	$plugins  = wp_list_pluck( $plugins, 'Version' );
	$nonce    = md5( serialize( $plugins ) );
	$headers  = secupress_get_basic_auth_headers();
	$args     = [ 'body' => [ 'items' => $plugins, 'type' => 'plugin', '_wpnonce' => $nonce ], 'headers' => $headers ];
	$response = wp_remote_post( SECUPRESS_API_MAIN . 'vulns/v2/', $args );
	if ( ! is_wp_error( $response ) && 200 === wp_remote_retrieve_response_code( $response ) ) {
		$response = wp_remote_retrieve_body( $response );
		// Store the results
		$response = json_decode( $response, true );
		if ( is_array( $response ) && count( $response ) >= 0 ) {
			$response = json_encode( $response );
			update_site_option( SECUPRESS_BAD_PLUGINS, $response );
			$dt = get_date_from_gmt( date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), time() ), get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
			secupress_set_option( 'bad_plugins_last_update', date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $dt ) ) );
		}
	}
	if ( ! wp_doing_cron() ) {
		wp_safe_redirect( wp_get_referer() );
	}
}

add_action( 'secupress.pro.plugins.deactivation',                                     'secupress_bad_plugins_deactivation' );
add_action( 'secupress.modules.deactivate_submodule_' . basename( __FILE__, '.php' ), 'secupress_bad_plugins_deactivation' );
/**
 * Remove the crons.
 *
 * @since 2.1
 * @author Julio Potier
 */
function secupress_bad_plugins_deactivation() {
	wp_clear_scheduled_hook( 'secupress_bad_plugins' );
	wp_clear_scheduled_hook( 'secupress_bad_plugins_maybe_do_checks' );
}

/* ============================================== */
/* ========== DISPLAY LAST UPDATE DATA ========== */
/* ============================================== */

add_action( 'load-plugins.php', 'secupress_display_late_updates_notice' );
/**
 * Display a notice if a plugin is not updated since a long time
 *
 * @since 2.5
 * @author Julio Potier
 */
function secupress_display_late_updates_notice() {
	$old_pl  = get_site_option( SECUPRESS_OLD_PLUGINS, [] );
	if ( ! $old_pl ) {
		return;
	}
	$updates  = get_plugin_updates();
	$delays   = [];
	foreach ( $updates as $plugin_file => $update_data ) {
		if ( isset( $old_pl[ $plugin_file ] ) ) {
			$days_old = time() - $old_pl[ $plugin_file ];
			if ( $days_old <= 4 * DAY_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$delays['1_info_notice-alt_notice-info_no-plugin-title_secupress-mini-notice'][ $plugin_file ]       = [ 'icon' => 'lightbulb', 'name' => $update_data->Name, 'ver' => $update_data->Version, 'since' => $old_pl[ $plugin_file ] ];
			} elseif ( $days_old <= 8 * DAY_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$delays['2_warning_notice-alt_notice-warning_no-plugin-title_secupress-mini-notice'][ $plugin_file ] = [ 'icon' => 'warning',   'name' => $update_data->Name, 'ver' => $update_data->Version, 'since' => $old_pl[ $plugin_file ] ];
			} elseif ( $days_old <= 2 * YEAR_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$delays['3_error_notice-alt_notice-error_no-plugin-title_secupress-mini-notice'][ $plugin_file ]     = [ 'icon' => 'flag',      'name' => $update_data->Name, 'ver' => $update_data->Version, 'since' => $old_pl[ $plugin_file ] ];
			} else {
				// Do not display any notice, we already do it
			}
		}
	}
	if ( ! empty( $delays ) ) {
		asort($delays);
		foreach ( $delays as $crit => $delay_data ) {
			$message  = '<ul>';
			foreach ( $delay_data as $_file => $_data ) {
				// Generate a unique nonce for this plugin to use as ID (same as WordPress uses for update links)
				$nonce = wp_create_nonce( 'upgrade-plugin_' . $_file );
				$nonce_id = 'secupress-notice-' . md5( $nonce );
				// translators: 1: plugin name, 2: plugin version, 3: time since update, 4: anchor link
				$message .= '<li id="' . esc_attr( $nonce_id ) . '" data-plugin-file="' . esc_attr( $_file ) . '" data-nonce-hash="' . esc_attr( md5( $nonce ) ) . '">' . sprintf( '%5$s ' . __( '<strong>%1$s v%2$s</strong> update waiting since <strong>%3$s</strong> <a href="#%4$s"><span class="dashicons dashicons-arrow-down-alt2"></span></a>', 'secupress' ), 
												$_data['name'], 
												$_data['ver'], 
												human_time_diff( time(), $_data['since'] ),
												dirname( $_file ) . '-update',
												'<span class="dashicons dashicons-' . $_data['icon'] . ' secupress-notice"></span>'
											) . '</li>';
			}
			$message .= '<ul>';
			secupress_add_notice( $message, $crit, false, 'update_plugins' );
		}
	}
}

add_action( 'admin_head', 'secupress_customize_admin_bar_updates_background' );
add_action( 'wp_head', 'secupress_customize_admin_bar_updates_background' );
/**
 * Customize the background color of the admin bar updates menu item.
 *
 * @since 2.5
 * @author Julio Potier
 */
function secupress_customize_admin_bar_updates_background() {
	if ( ! is_admin_bar_showing() ) {
		return;
	}
	$old_pl  = get_site_option( SECUPRESS_OLD_PLUGINS, [] );
	if ( ! $old_pl ) {
		return;
	}
	if ( ! function_exists ( 'get_plugin_updates' ) ) {
		require_once( ABSPATH . '/wp-admin/includes/update.php' );
	}
	$updates  = get_plugin_updates();
	$color    = '';
	$colors   = [ 'info' => '#26B3A9', 'warning' => '#F7AB13', 'error' => '#F2295E' ];
	$found    = [ 'info' => 0, 'warning' => 0, 'error' => 0 ];
	foreach ( $updates as $plugin_file => $update_data ) {
		if ( isset( $old_pl[ $plugin_file ] ) ) {
			$days_old = time() - $old_pl[ $plugin_file ];
			if ( $days_old <= 4 * DAY_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$found['info']++;
			} elseif ( $days_old <= 8 * DAY_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$found['warning']++;
			} elseif ( $days_old <= 2 * YEAR_IN_SECONDS && $days_old >= 1 * DAY_IN_SECONDS ) {
				$found['error']++;
			}
		}
	}
	if ( $found['error'] > 0 ) {
		$color = $colors['error'];
	} elseif ( $found['warning'] > 0 ) {
		$color = $colors['warning'];
	} elseif ( $found['info'] > 0 ) {
		$color = $colors['info'];
	}
	if ( $color ) {
		echo '<style type="text/css">
			#wpadminbar #wp-admin-bar-updates {
				background-color: ' . $color . ' !important;
			}
		</style>';
	}
}
