<?php
/**
 * Module Name: Bad Email Domains
 * Description: Prevent users to use trash, temporary, bad known domains or fake MX records to create an account
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

add_filter( 'registration_errors', 'secupress_pro_bad_email_domain_validate_user_registration', 10, 3 );
/**
 * Hook the secupress_pro_bad_email_domain_mx_domain_exists() function to user registration and validate email domain.
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param WP_Error $errors Registration errors.
 * @param string $sanitized_user_login The sanitized user login.
 * @param string $user_email The user email address.
 * 
 * @return WP_Error|bool Modified registration errors or true if the email domain is allowed.
 */
function secupress_pro_bad_email_domain_validate_user_registration( $errors, $sanitized_user_login, $user_email ) {
	/**
	 * Handle the errors from registration
	 * 
	 * @param (WP_Error) $errors
	 * @param (string) $sanitized_user_login
	 * @param (string) $user_email
	 * 
	 * @since 2.2.6
	 * 
	 * @return (WP_Error) $errors
	 * */
	$errors  = apply_filters( 'secupress.plugins.bad_email_domains.errors', $errors, $sanitized_user_login, $user_email );
	/**
	 * Handle the errors from registration
	 * 
	 * @param (string) $message
	 * @param (string) $sanitized_user_login
	 * @param (string) $user_email
	 * 
	 * @since 2.2.6
	 * 
	 * @return (string) $message
	 * */
	$message = apply_filters( 'secupress.plugins.bad_email_domains.message', __( '<strong>Error</strong>: User registration is currently not allowed.', 'secupress' ), $errors, $sanitized_user_login, $user_email );

	if ( secupress_pro_bad_email_domain_is_bad( $user_email ) ) {
		secupress_log_attack( 'users' );
		$errors->add( 'registerfail', $message ); // We stop it, add error, KO for us
	}
	return $errors;
}

/**
 * Check if the email is a bad one from our domain tests
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $email
 * @return (bool)
 **/
function secupress_pro_bad_email_domain_is_bad( $email ) {
	$domain  = substr( strrchr( $email, '@' ), 1 );

	// Check if the user email domain TLD is restricted
	$restricted_tlds = secupress_pro_bad_email_domain_get_bad_tld_for_email();
	$user_tld	     = strrchr( $domain, '.' );
	if ( in_array( $user_tld, $restricted_tlds, true ) ) {
		return true;
	}
	// Check if the email is directly in our email list known from malwares (not only domain, the whole address).
	if ( secupress_pro_bad_email_domain_get_domain_status( $email, 'bad' ) ) {
		return true;
	}
	// Check if the email domain exists and is valid using secupress_pro_bad_email_domain_mx_domain_exists()
	if ( ! secupress_pro_bad_email_domain_mx_domain_exists( $domain ) ) {
		return true;
	}
	// Check if the email domain is in the good domains list and allowed
	if ( secupress_pro_bad_email_domain_get_domain_status( $domain, 'good' ) ) {
		return false; // We let go, it's ok for us
	}
	// Check if the email domain is in the bad domains list and not allowed
	if ( secupress_pro_bad_email_domain_get_domain_status( $domain, 'bad' ) ) {
		return true;
	}
	return false;
}

/**
 * Check if the email is a bad one from our tld tests
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $email
 * @return (bool)
 **/
function secupress_pro_bad_email_domain_get_bad_tld_for_email() {
	return apply_filters( 'secupress.plugins.bad_email_domains.tld', [ '.tk', '.ml', '.ga', '.cf', '.gq' ] );
}

/**
 * Check if a domain is in the good domains list and allowed.
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $domain
 * @param (string) $list 'good' or 'bad'
 * 
 * @return bool True if the domain is in the good domains list and allowed, false otherwise.
 */
function secupress_pro_bad_email_domain_get_domain_status( $domain, $list ) {
	global $wpdb;
	static $good_list, $bad_list;

	$domain   = strtolower( $domain );
	/**
	 * Shortcut for another domain
	 * 
	 * @since 2.2.6
	 * @param (string) $domain
	 * @param (string) $list
	 * 
	 * @return (bool) $list 'good' + $return true = good domain
	 */
	$return = apply_filters( 'secupress.plugins.bad_email_domains.allowed', null, $domain, $list );
	if ( is_bool( $return ) ) {
		return $return;
	}

	if ( 'bad' === $list && isset( $bad_list ) ) {
		return isset( $bad_list[ $domain ] );
	}

	if ( 'good' === $list && isset( $good_list ) ) {
		return isset( $good_list[ $domain ] );
	}

	$list      = $list === 'good' ? 'good' : 'bad';
	$filename  = secupress_get_data_file_path( $list . '_email_domains' );
	$_list     = [];
	if ( file_exists( $filename ) ) {
		$_list = explode( "\n", file_get_contents( $filename ) );
	}
	$_list      = array_flip( $_list );

	return isset( $_list[ $domain ] );
}

/**
 * Check if an MX record exists for a domain.
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (string) $domain The domain to check.
 * 
 * @return (bool) True if MX record exists, false otherwise or if the verification cannot be performed.
 */
function secupress_pro_bad_email_domain_mx_domain_exists( $domain ) {
	// Check if the result is already cached
	static $cache = array();

	if ( isset( $cache[ $domain ] ) ) {
		return $cache[ $domain ];
	}
	$parsed_domain     = parse_url( $domain, PHP_URL_HOST );
	if ( ! $parsed_domain ) {
		$parsed_domain = parse_url( 'https://' . $domain, PHP_URL_HOST );
	}
	$domain            = $parsed_domain;
	$domain_dot        = rtrim( $domain, '.' ) . '.';

	// If unable to perform check, can't say no! :/
	$result            = true;

	// Perform the DNS lookup
	if ( function_exists( 'checkdnsrr' ) ) {
		// Use checkdnsrr() if available
		$result  = checkdnsrr( $domain_dot, 'MX' );
	} elseif ( function_exists( 'gethostbyname' ) ) {
		// Use gethostbyname() if available
		$record  = gethostbyname( $domain );
		$result  = $record !== $domain;
	} elseif ( function_exists( 'dns_get_record' ) ) {
		// Use dns_get_record() if available
		$records = dns_get_record( $domain, DNS_MX );
		$result  = ! empty( $records );
	} elseif ( function_exists( 'shell_exec' ) ) {
		// Use shell_exec() if available
		$command = sprintf( 'nslookup -type=MX %s', $domain_dot );
		$output  = shell_exec($command);
		if ( $output ) {
			$pattern = '/^' . preg_quote( $domain_dot ) . '.*\bMX\b/im';
			$matches = preg_match( $pattern, $output );
			$result  = $matches > 0;
		}
	}
	
	/**
	 * Shortcut to validate a MX domain
	 * 
	 * @since 2.2.6
	 * 
	 * @param (string) $result
	 * @param (string) $domain
	 * 
	 * @return (string)
	 */
	$result           = apply_filters( 'secupress.plugins.bad_email_domains.mx', $result, $domain );
	// Cache the result
	$cache[ $domain ] = $result;

	return $result;
}
