<?php
/**
 * Module Name: GeoIP Login
 * Description: Challenge the login when the GeoIP does not match the previous ones.
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

// EMERGENCY BYPASS!
if ( defined( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) && SECUPRESS_ALLOW_LOGIN_ACCESS ) {
	return;
}

define( 'SECUPRESS_GEOIP_LOGIN_', 'secupress_geoip_login_' );

add_action( 'authenticate', 'secupress_pro_geoip_on_login', SECUPRESS_INT_MAX - 2, 3 );
/**
 * Confront logins on bad geoip
 *
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (mixed) $raw_user
 * @param (string) $username
 * @param (string) $password The users's password in clear
 * 
 * @return (mixed) $raw_user Can also display a form to ask a code
 **/
function secupress_pro_geoip_on_login( $raw_user, $username, $password ) {
	if ( 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
		return $raw_user;
	}

	$user_id        = secupress_is_user( $raw_user ) ? $raw_user->ID : 0;
	// We may need to switch to user locale now
	if ( $user_id ) {
		switch_to_locale( get_user_locale( $user_id ) );
	}
	if ( ! is_a( $raw_user, 'WP_User' ) ) {
		$tmp_user   = secupress_get_user_by( $username );
		if ( ! secupress_is_user( $tmp_user ) ) {
			return $raw_user; // Wrong password return is here
		}
		$user_token = secupress_pro_geoip_get_user_option( 'token', $tmp_user->ID );
		$user_token = ! $user_token ? '' : $user_token;
		$user_db    = secupress_pro_geoip_get_user_from_meta_value( 'token', $password );
		$time       = secupress_pro_geoip_get_user_option( 'timeout', $tmp_user->ID );
		if ( ! secupress_is_user( $tmp_user ) || ! $user_db || $user_db !== $tmp_user->ID || ! hash_equals( $user_token, $password ) ) {
			return $raw_user; // Wrong token return is here
		}

		if ( $time < time() ) {
			$error  = __( 'Too much time has elapsed between the first step and now.', 'secupress' );
			return new WP_Error( 'expired_time', $error );
		}

		$raw_user   = $tmp_user;
		$user_id    = $raw_user->ID;
		// We may need to do it here now.
		switch_to_locale( get_user_locale( $user_id ) );
		unset( $tmp_user );
	}

	$user_locations = secupress_pro_geoip_get_user_option( 'locations', $user_id );
	switch_to_locale( get_user_locale( $user_id ) );
	$user_ip        = secupress_get_ip();
	$device         = secupress_get_device_infos();
	$devices        = secupress_pro_geoip_get_user_option( 'devices', $user_id );
	$device_ok      = ! $devices || isset( $devices[ $device['sign'] ] );
	$device_ok      = $device_ok ?: ( ! (bool) secupress_get_module_option( 'login-protection_geoip_login_device', '0', 'users-login' ) );
	$device_ok      = secupress_is_expert_mode() ? $device_ok : true; // On non expert mode, do not check;
	$challenge_mode = secupress_get_module_option( 'login-protection_geoip_login_mode', 'ip-smooth', 'users-login' );
	$challenge_mode = secupress_is_expert_mode() ? $challenge_mode : 'ip-smooth'; // On non expert mode, fallback to default

	if ( $device_ok && $user_locations ) {
		foreach ( $user_locations as $data ) {
			switch( $challenge_mode ) {
				case 'ip-strict': // by ipAddress strict
					if ( 0 === strcmp( $user_ip, $data['ipAddress'] ) ) {
						return $raw_user;
					}
				break;

				default;
				case 'ip-smooth': // by ipAddress, but smoother, close in the same network
					if ( secupress_ips_are_close( $user_ip, $data['ipAddress'] ) ) {
						return $raw_user;
					}
				break;
			}
		}
	}

	if ( ! isset( $user_locations[ $user_ip ] ) ) {
		if ( false === ( $body = get_transient( SECUPRESS_GEOIP_LOGIN_ . md5( $user_ip ) ) ) ) {
			/**
			 * Filter the API URL to use for the geoip login.
			 *
			 * @since 2.6
			 * @author Julio Potier
			 * 
			 * @param (string) $api_url The API URL to use, must contain %s for the IP address.
			 * @param (string) $user_ip The user IP address.
			 * 
			 * @return (string) The API URL to use.
			 */
			$api_url  = apply_filters( 'secupress.plugins.geoip_login.api_url', 'https://free.freeipapi.com/api/json/%s' ); // Commercial use allowed
			$api_url  = sprintf( $api_url, $user_ip );
			$headers  = [
				'User-Agent' => 'SecuPressPro/' . SECUPRESS_VERSION,
				// 'Authorization' => 'Bearer TOKEN', // DO not use in free version.
			];
			// Need more checks? https://freeipapi.com/docs/api-reference/api-introduction
			/**
			 * Filter the headers to use for the geoip login.
			 *
			 * @since 2.6
			 * @author Julio Potier
			 * 
			 * @param (array) $headers The headers to use.
			 * @param (string) $user_ip The user IP address.
			 * 
			 * @return (array) The headers to use.
			 */
			$headers  = apply_filters( 'secupress.plugins.geoip_login.headers', $headers, $user_ip );
			// We do a distant call instead of using the local database to get the geoip data. On login, lose a small amount af time is ok.
			$response = wp_remote_get( $api_url, [ 'headers' => $headers ] );
			// Can't do anything...
			if ( is_wp_error( $response ) ) {
				return $raw_user;
			}
			$body     = wp_remote_retrieve_body( $response );
			/**
			 * Filter the body of the geoip login.
			 *
			 * @since 2.6
			 * @author Julio Potier
			 * 
			 * @param (array) $body The body of the geoip login.
			 * @param (string) $body The raw body of the geoip login.
			 * @param (string) $user_ip The user IP address.
			 * 
			 * @return (array) The body of the geoip login.
			 */
			$body     = apply_filters( 'secupress.plugins.geoip_login.body', json_decode( $body, true ), $body, $user_ip );
			set_transient( SECUPRESS_GEOIP_LOGIN_ . md5( $user_ip ), $body, 10 * MINUTE_IN_SECONDS );
		}
	} else {
		$body         = $user_locations[ $user_ip ];
	}
	if ( ! $user_locations ) { // First login for this user since module activated
		if ( isset( $body['ipVersion'] ) ) {
			secupress_pro_geoip_update_user_data( 'locations', $body, $user_id );
			secupress_pro_geoip_update_user_device( 'devices', $device, $user_id );
		}
		return $raw_user;
	} else {
		if ( $device_ok ) {
			foreach ( $user_locations as $data ) {
				switch( $challenge_mode ) {
					case 'city': // by cityName
						if ( 0 === strcmp( $body['cityName'], $data['cityName'] ) ) {
							return $raw_user;
						} 
					break;

					case 'region': // by regionName
						if ( 0 === strcmp( $body['regionName'], $data['regionName'] ) ) {
							return $raw_user;
						} 
					break;

					case 'country': // by countryName
						if ( 0 === strcmp( $body['countryName'], $data['countryName'] ) ) {
							return $raw_user;
						} 
					break;

				}
			}
		}
	}
	global $sp_action;
	$sp_action        = 'challenge';
	$post_param       = "secupress-geoip-login-{$sp_action}";
	$nonce_action     = "{$post_param}-{$user_id}";
	$error            = '';

	// Role is not affected by this module
	if ( ! secupress_is_affected_role( 'users-login', 'login-protection_geoip_login', $raw_user ) ) {
		secupress_pro_geoip_delete_user_option( 'token', $raw_user->ID );
		secupress_pro_geoip_delete_user_option( 'timeout', $raw_user->ID );
		secupress_pro_geoip_delete_user_option( 'code', $raw_user->ID );
		secupress_pro_geoip_delete_user_option( 'devices', $raw_user->ID );
		secupress_pro_geoip_delete_user_option( 'locations', $raw_user->ID );
		return $raw_user;
	}

	$time             = secupress_pro_geoip_get_user_option( 'timeout', $user_id );
	$new_token        = secupress_pro_geoip_get_user_option( 'token', $user_id );
	if ( $time < time() ) {
		$new_token    = wp_hash( wp_generate_password( 32, false ) );
		secupress_pro_geoip_update_user_option( 'token', $new_token, $user_id );
		secupress_pro_geoip_update_user_option( 'timeout', time() + ( 10 * MINUTE_IN_SECONDS ), $user_id );
		$user_code    = secupress_generate_key( 4, '1234567890' ); // only digits, not a choice.
		secupress_pro_geoip_update_user_option( 'code', $user_code, $user_id );
		$subject      = sprintf( __( '[###SITENAME###] %s will confirm your identity.', 'secupress' ), $user_code );
		$subject      = apply_filters( 'secupress.plugins.geoip_login.subject', $subject, $user_code );
		$message      = sprintf(
		__( 'Hello ###USERNAME###,

Use [%s] to confirm you identity on ###SITEURL###:

You can safely ignore and delete this email if you do not want to take this action.

Regards,
All at ###SITENAME###
###SITEURL###', 'secupress' ),
		esc_html( $user_code )
	);
		$message      = str_replace( '###USERNAME###', esc_html( $raw_user->display_name ), $message );
		$message      = apply_filters( 'secupress.plugins.geoip_login.message', $message, $user_code, $raw_user );
		secupress_send_mail( $raw_user->user_email, $subject, $message );
	}
	// A code is submitted.
	if ( isset( $_POST['code'], $_POST['sp_action'] ) && $sp_action === $_POST['sp_action'] ) {

		if ( empty( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], $nonce_action ) ) {
			secupress_die( __( 'Something went wrong.', 'secupress' ), '', [ 'force_die' => true, 'context' => 'location', 'attack_type' => 'login' ] );
		}

		$code         = substr( preg_replace( '/\D/', '', $_POST['code'] ), 0, 4 );
		$user_code    = secupress_pro_geoip_get_user_option( 'code', $user_id );
		if ( empty( $code ) ) {
			$error    = __( 'Code required', 'secupress' );
		} elseif( ! hash_equals( $user_code, $code ) ) {
			$error    = __( 'Code does not match.', 'secupress' );
		} else {
			if ( isset( $_POST['remember_ip'], $body['ipVersion'] ) && 
				( ( 'ip-strict' === $challenge_mode && ! isset( $user_locations[ $user_ip ] ) ) ||
					( 'ip-smooth' === $challenge_mode && empty( array_filter( array_map( 'secupress_ips_are_close', array_keys( $user_locations ) ) ) ) ) )
 			) {
				secupress_pro_geoip_update_user_data( 'locations', $body, $user_id );
				delete_transient( SECUPRESS_GEOIP_LOGIN_ . md5( $user_ip ) );
			}
			if ( isset( $_POST['remember_device'], $body['ipVersion'] ) && ! $device_ok && false === stripos( $device['raw_ua'], 'unknown' ) ) {
				secupress_pro_geoip_update_user_device( 'devices', $device, $user_id );
			}
			secupress_pro_geoip_delete_user_option( 'token', $user_id );
			secupress_pro_geoip_delete_user_option( 'timeout', $user_id );
			secupress_pro_geoip_delete_user_option( 'code', $user_id );
			return $raw_user;
		}
	}

	ob_start();
	$message          = __( '<strong>New Location Detected</strong><br>Please enter the verification code sent to your mailbox to confirm your identity.', 'secupress' );
	$wp_errors        = new WP_Error();
	$wp_errors->add( "{$sp_action}-message", $message, 'message' );
	if ( $error ) {
		$wp_errors->add( "{$sp_action}-error", $error, 'error' );
		add_action( 'secupress_login_page.shake_js', '__return_true' );
	}
	// The form.
	ob_start();
	?>
	<form name="loginform" id="loginform" action="" method="post">
		<div id="password" class="secupress-user-pass1-wrap user-pass-wrap">
			<div class="wp-pwd">
				<h2><?php _e( 'New Location Detected', 'secupress' ); ?></h2>
				<ul>
					<li><?php printf( '<strong>%s</strong>, %s (%s)', esc_html( $body['cityName'] ), esc_html( $body['regionName'] ), esc_html( $body['countryName'] ) . ' ' . secupress_get_flag( $body['countryCode'] ) ); ?></li>
					<li><?php printf( '<strong>%s</strong> (%s)', esc_html( $body['ipAddress'] ), esc_html( $body['asnOrganization'] ) ); ?></li>
					<li><?php printf( '<strong>%s</strong>, %s (%s)', esc_html( $device['browser'] ), esc_html( $device['os'] ), esc_html( $device['device'] ) ); ?></li>
				</ul>
				<?php
				$map_url = esc_url_raw( 'https://facilmap.org/?search=false#15/' . esc_html( $body['latitude'] ) . '/' . esc_html( $body['longitude'] ) . '/TrTo/' );
				printf( __( '%sOpen a map%s to check the location', 'secupress' ), '<a href="' . $map_url . '" target="_blank">', '</a>' );
				?>
				<p>
					<span class="password-input-wrapper">
						<input type="text" name="code" maxlength="4" pattern="\d{4}" id="secupress-code" class="input regular-text" value="" autocomplete="off" />
					</span>
				</p>
				<p class="forgetmenot">
					<?php
					if ( 
						( 'ip-strict' === $challenge_mode && ! isset( $user_locations[ $user_ip ] ) ) ||
						( 'ip-smooth' === $challenge_mode && empty( array_filter( array_map( 'secupress_ips_are_close', array_keys( $user_locations ) ) ) ) ) ||
						( 'city'      === $challenge_mode && ! isset( array_flip( wp_list_pluck( $user_locations, 'cityName' ) )[ $body['cityName'] ] ) ) ||
						( 'region'    === $challenge_mode && ! isset( array_flip( wp_list_pluck( $user_locations, 'regionName' ) )[ $body['regionName'] ] ) ) ||
						( 'country'   === $challenge_mode && ! isset( array_flip( wp_list_pluck( $user_locations, 'countryName' ) )[ $body['countryName'] ] ) )
						) { 
							?>
						<label>
							<input type="checkbox" name="remember_ip" id="secupress-remember-ip" value="1" checked="checked" />
							<?php _e( 'Save this location as trusted.', 'secupress' ); ?>
						</label>
						<?php
					}
					if ( ! $device_ok && false === stripos( $device['raw_ua'], 'unknown' ) ) { 
						?>
						<label>
							<input type="checkbox" name="remember_device" id="secupress-remember-device" value="1" checked="checked" />
							<?php _e( 'Save this device as trusted.', 'secupress' ); ?>
						</label>
						<?php 
					}
					?>
				</p>
			</div>
		</div>
		<p class="submit">
			<?php
			if ( ! function_exists( 'submit_button' ) ) {
				require_once ABSPATH . 'wp-admin/includes/template.php';
			}
			submit_button( _x( 'Confirm', 'verb', 'secupress' ), 'primary', 'submit' );
			wp_nonce_field( $nonce_action );
			?>
		</p>
		<input type="hidden" name="log" value="<?php echo esc_attr( $username ); ?>" />
		<input type="hidden" name="pwd" value="<?php echo esc_attr( $new_token ); ?>" />
		<input type="hidden" name="sp_action" value="<?php echo esc_attr( $sp_action ); ?>">
	</form>
	<style>
	#loginform .wp-pwd ul {
		word-wrap: break-word;
		overflow-wrap: break-word;
		word-break: break-word;
	}
	</style>
	<script>
	document.getElementById('secupress-code').addEventListener('input', function(e) {
		this.value = this.value.replace(/\D/g, '');
		if (this.value.length > 4) {
			this.value = this.value.slice(0, 4);
		}
	});
	</script>
	<?php

	$content = ob_get_contents();
	ob_end_clean();
	unset( $raw_user );
	if ( ! secupress_is_soft_request() ) {
		wp_send_json_error( [ 'error' => $error, 'message' => $message ] );
	} else {
		secupress_login_page( __( 'Please confirm you identity', 'secupress' ), $content, $wp_errors, $user_id );
	}
}

/**
 * Get a geoip locations user option easily
 *
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_geoip_get_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$uid          = $uid ?: $current_user->ID;
	return get_user_option( SECUPRESS_GEOIP_LOGIN_ . $option, $uid );
}

/**
 * Set a device user option easily
 * 
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (array) $device
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_geoip_update_user_device( $option, $device, $uid = 0 ) {
	if ( ! isset( $device['raw_ua'] ) ) {
		return false;
	}
	$current_user   = wp_get_current_user();
	$uid            = $uid ?: $current_user->ID;
	$options        = secupress_pro_geoip_get_user_option( $option, $uid );
	$options        = $options ?: [];
	$device['time'] = time();
	
	$options[ $device['sign'] ] = $device;
	return update_user_option( $uid, SECUPRESS_GEOIP_LOGIN_ . $option, $options );
}

/**
 * Set a geoip locations user option easily
 * 
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (array) $data
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_geoip_update_user_data( $option, $data, $uid = 0 ) {
	if ( ! isset( $data['ipAddress'] ) ) {
		return false;
	}
	$current_user = wp_get_current_user();
	$uid          = $uid ?: $current_user->ID;
	$options      = secupress_pro_geoip_get_user_option( $option, $uid );
	$options      = $options ?: [];
	$data         = [
					'ipAddress'       => $data['ipAddress'], 
					'latitude'        => $data['latitude'], 
					'longitude'       => $data['longitude'], 
					'cityName'        => $data['cityName'], 
					'regionName'      => $data['regionName'], 
					'countryName'     => $data['countryName'], 
					'countryCode'     => $data['countryCode'], 
					'asnOrganization' => $data['asnOrganization'], 
				];
	$data['sign'] = md5( serialize( $data ) );
	$data['time'] = time();
	
	$options[ $data['ipAddress'] ] = $data;
	return update_user_option( $uid, SECUPRESS_GEOIP_LOGIN_ . $option, $options );
}

/**
 * Get a geoip login option name
 * 
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * 
 * @return (string)
 **/
function secupress_pro_geoip_get_option_name( $option ) {
	global $wpdb;
	return sprintf( '%s' . SECUPRESS_GEOIP_LOGIN_ . '%s', $wpdb->prefix, $option );
}

/**
 * Get a user_id from a meta value
 *
 * @since 2.6
 * @author Julio Potier
 *
 * @param (string) $key
 * @param (string) $value
 *
 * @return (array)
 */
function secupress_pro_geoip_get_user_from_meta_value( $key, $value ) {
	global $wpdb;
	$res = secupress_cache_data( "$key|$value" );
	if ( $res ) {
		return $res;
	}
	$res = (int) $wpdb->get_var( $wpdb->prepare( "SELECT user_id FROM $wpdb->usermeta WHERE meta_key = %s AND meta_value = %s", secupress_pro_geoip_get_option_name( $key ), $value ) );
	secupress_cache_data( "$key|$value", $res );
	return $res;
}

/**
 * Set a geoip options user easily
 * 
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (array) $data
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_geoip_update_user_option( $option, $value, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$uid          = $uid ?: $current_user->ID;
	return update_user_option( $uid, SECUPRESS_GEOIP_LOGIN_ . $option, $value );
}

/**
 * Delete a geo login user option easily
 * 
 * @since 2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_geoip_delete_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$uid          = $uid ?: $current_user->ID;
	return delete_user_option( $uid, SECUPRESS_GEOIP_LOGIN_ . $option );
}
