<?php
/*
Module Name: OTP Authentication
Description: Two-Factor Authentication using an app as OTP (One Time Password) generator.
Main Module: users-login
Author: SecuPress
Version: 2.2.6
*/
//// sprintf( __( 'Invalid Link.<br>Please try to %slog in again%s.', 'secupress' ), '<a href="' . wp_login_url( '', true ) . '">', '</a>' )
defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

// EMERGENCY BYPASS!
if ( defined( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) && SECUPRESS_ALLOW_LOGIN_ACCESS ) {
	return;
}
return;
add_action( 'admin_init', 'secupress_otpauth_register_css_js' );
add_action( 'login_head', 'secupress_otpauth_register_css_js' );
/**
 * Register CSS and JS files
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_register_css_js() {
	$suffix  = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	$version = $suffix ? SECUPRESS_VERSION : time();
	wp_register_script( 'secupress-jquery-qrcode', SECUPRESS_PRO_MODULES_URL . 'users-login/plugins/inc/js/jquery.qrcode' . $suffix . '.js', [ 'jquery' ], $version );
	wp_register_script( 'secupress-qrcode',        SECUPRESS_PRO_MODULES_URL . 'users-login/plugins/inc/js/qrcode' . $suffix . '.js',        [ 'jquery' ], $version );
	wp_register_style(  'secupress-otp-auth',      SECUPRESS_PRO_MODULES_URL . 'users-login/plugins/inc/css/otp-auth' . $suffix . '.css',    [],           $version );
}

add_action( 'login_head', 'secupress_otpauth_css_obs' );
/**
 * Add CSS rules in login head + start buffer to replace DOM elements
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @return (string)
 **/
function secupress_otpauth_css_obs() {
	global $action;
	wp_enqueue_style( 'secupress-otp-auth' );
	?>
	<style type="text/css">
	.login h1 a{ background-image: url('<?php echo esc_js( get_site_icon_url( 84, includes_url( 'images/w-logo-blue-white-bg.png' ) ) ); ?>'); border-radius: 100%; border: 1px solid #c3c4c7; box-shadow: 0 3px 6px rgba(0, 0, 0, 0.2) }
	</style>
	<?php
	if ( 'login' === $action ) {
		ob_start( 'secupress_otpauth_hide_password_field_ob' );
	}
}

add_filter( 'lostpassword_post', 'secupress_otpauth_prevent_password_reset_for_otp_validated_users', 10, 2 );
/**
 * An OTP activated user cannot reset its password.
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_prevent_password_reset_for_otp_validated_users( $errors, $userdata ) {
	if (    secupress_is_user( $userdata ) && 
			secupress_is_affected_role( 'users-login', 'double-auth', $userdata ) &&
			secupress_otpauth_get_user_option( 'secret', $userdata->ID ) &&
			secupress_otpauth_get_user_option( 'verified', $userdata->ID )
	) {
        $errors->add( 'no_password_reset', __( 'Password reset is not allowed for this user.', 'secupress' ) );
    }

    return $errors;
}

add_action( 'login_form_rp',        'secupress_otpauth_prevent_password_reset_for_validated_users_on_registration' );
add_action( 'login_form_resetpass', 'secupress_otpauth_prevent_password_reset_for_validated_users_on_registration' );
/**
 * If the user id still pending, they cannot ask for a reset password
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_prevent_password_reset_for_validated_users_on_registration() {
	if ( ! isset( $_GET['key'], $_GET['login'] ) ) {
		return;
	}
	$userdata = secupress_get_user_by( $_GET['login'] );
	if (    secupress_is_user( $userdata ) && 
			secupress_is_affected_role( 'users-login', 'double-auth', $userdata ) &&
			secupress_otpauth_get_user_option( 'secret', $userdata->ID ) &&
			secupress_otpauth_get_user_option( 'verified', $userdata->ID )
	) {
		login_header( __( 'Password Reset', 'secupress' ), '<p class="notice notice-error message reset-pass">' . __( 'Password reset is not allowed for this user.', 'secupress' ) . '</p>' );
		login_footer();
		die();
	}	
}

add_filter( 'login_headertext', 'secupress_otpauth_login_headertext', 1 );
/**
 * Tweak the description text on login header
 * Low prio
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_login_headertext( $login_header_text ) {
	return get_bloginfo( 'description' );
}

add_filter( 'login_headerurl', 'secupress_otpauth_login_headerurl', 1 );
/**
 * Tweak the URL text on login header
 * Low prio
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_login_headerurl( $login_header_url ) {
	return home_url();
}
/**
 * Alter the buffer content to hide the password label and field.
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (string) $buffer Contains the login page between the action `login_head` and `login_footer`.
 *
 * @return (string)
 */
function secupress_otpauth_hide_password_field_ob( $buffer ) {
	global $errors;
	echo 'foobar';
	$sp_roles = secupress_get_module_option( 'double-auth_affected_role', [], 'users-login' );
	$buffer   = str_replace( '<div class="user-pass-wrap">', '<div class="user-pass-wrap hide-if-js"><input type="hidden" name="need_otpauth" value="1" />', $buffer );
	$buffer   = str_replace( 'value="' . esc_attr__( 'Log In' ) . '"', 'data-origvalue="' . esc_attr__( 'Log In' ) . '" data-newvalue="' . esc_attr__( 'Continue', 'secupress' ) . '" value="' . esc_attr__( 'Log In' ) . '"', $buffer );
	$replace  = '<p class="input" id="fake-input-p"></p><span id="fake-input-s"><button id="fake-input-switch" type="button" class="button-small button-link">' . __( 'Not you? Switch account.', 'secupress' ) . '</button></span>';
	$buffer   = str_replace( '<label for="user_login">', $replace . '<label for="user_login">', $buffer );
	$log      = isset( $_POST['log'] ) ? secupress_get_user_by( $_POST['log'] ) : '';
	return $buffer;
}

add_action( 'login_form_login', 'secupress_otpauth_lostpassword_flag', 20 );
/**
 * Send the magic link to a user
 * 
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_otpauth_lostpassword_flag() {
	global $errors;
	if ( ! isset( $_POST['magic_link'], $_POST['log'] ) ) {
		return;
	}
	$user = secupress_get_user_by( $_POST['log'] );
	if ( ! secupress_is_user( $user ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}
	if ( secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		echo '<div class="notice-error notice">' . __( 'You cannot login using a Magic Link. Please use your OTP App instead.', 'secupress' ) . '</div>';
	} else {
		secupress_passwordless_send_link( $user );
		wp_redirect( add_query_arg( 'action', 'passwordless_autologin', wp_login_url( '', true ) ) );
		die();
	}
}

add_action( 'login_footer', 'secupress_otpauth_footer_js_buffer', 1000 );
/**
 * End the buffer if we are on the login page
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_footer_js_buffer() {
	global $action;
	if ( 'otpauth_need_mobile' === $action ) {
	?>
	<script>
	jQuery( document ).ready( function($) {
		$('.login h1').css('top', '40px' );
	});
	</script>
	<?php
		return;
	}

	if ( 'otp_needed' === $action ) {
	?>
	<script>
	jQuery( document ).ready( function($) {
		if ( $('#loginform').length ) {
			$('.login-action-otp_needed h1').css('top', '40px' );
		}
		$('div.notice').insertAfter('p.submit');
		$('#fake-input-switch').on('click', function() {
			$('#login_error').remove();
			if (typeof localStorage !== 'undefined') {
				localStorage.removeItem('secupress_LS_login_data');
			}
			window.location.href = '<?php echo wp_login_url( '', true ); ?>';
		});		
	});
	</script>
	<?php
		return;
	}

	if ( 'login' !== $action ) {
		return;
	}
	?>
	<script>
	jQuery( document ).ready( function($) {
		$('div.notice').insertAfter('p.submit');
		$('#magic-link').on('click', function(e){
			$('.user-pass-wrap').remove();
		});
		$('#form_alt').insertAfter('#loginform');
		$('#user_login').on('keyup',
			function() {
				let val = $('#user_login').val().toLowerCase();
				let emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
				if ( emailRegex.test( val ) ) { // Works only for email logins.
					$('#user_login').css('text-transform','lowercase');
					let url = _wpUtilSettings.ajax.url + '?action=md5&e=' + val;
					let ajax = $.getJSON( url )
					.always( function(data) {
						$('.login h1 a').css({ "background-image": 'url(https://0.gravatar.com/avatar/' + data.responseText + '?s:180)', "border-radius": '100%'});
					});
				} else {
					$('#user_login').css('text-transform','');
					$('.login h1 a').css({ "background-image": ''});
				}
			}
		).trigger('keyup');

		$('.hide-if-js').hide();
		$('#wp-submit').val($('#wp-submit').data('newvalue'));
		function secupressContinueHandler() {
			$('.user-pass-wrap').show().find('input').prop('disabled', false).trigger('focus');
			$('#wp-submit').val($('#wp-submit').data('origvalue')).css('margin-bottom', '0px');
		}

		function secupressOTPHandler(e) {
			if (!$('.user-pass-wrap:visible').length) {
				e.preventDefault();
			}

			let val = $('#user_login').val();
			let url = _wpUtilSettings.ajax.url + '?action=otp_status&l=' + val;

			$.getJSON(url)
			.always(function(response) {
				if (!response || response.success === undefined) {
					return;
				}

				if (response.success) {
					$('.user-pass-wrap').hide();
					$('#wp-submit').closest('form').submit();
				} else {
					secupressContinueHandler();
					if (response.data !== 0) {
						$('.login h1').css('top', '40px');
						$('.login h1 a').css({
							"background-image": 'url(https://0.gravatar.com/avatar/' + response.data.gravatar + '?s:180)',
							"border-radius": '100%'
						});
						$('label[for="user_login"], #user_login').hide();
						$('#fake-input-p').text(('<?php echo esc_js( __( 'Howdy, %s', 'secupress' ) ) ?>.').replace('%s', response.data.display_name)).show();
						$('#fake-input-s').show();
					}
				}
			});
		}
		$('#loginform').on('submit', function(e){
			if ( typeof localStorage !== 'undefined' && $('#rememberme:checked:visible').length ) {
				let login    = $('#user_login').val();
				var toStore  = {'login':login};
				if ( login.length ) {
					localStorage.setItem('secupress_LS_login_data', JSON.stringify(toStore) );
				} else {
					localStorage.removeItem('secupress_LS_login_data');
				}
			}
		});
		$('#fake-input-switch').on('click', function() {
			$('#login_error, #form_alt').remove();
			if (typeof localStorage !== 'undefined') {
				localStorage.removeItem('secupress_LS_login_data');
			}
			$(this).parent().show();
			$('.login h1').css('top', '0px');
			$('label[for="user_login"], #user_login, .forgetmenot').show();
			$('#rememberme').prop('checked', false);
			$('#fake-input-s, #fake-input-p, .user-pass-wrap').hide();
			$('.login h1 a').css({
				"background-image": ''
			});
			$('#user_login').trigger('select').trigger('focus');
			$('#wp-submit').val($('#wp-submit').data('newvalue'));
		});

		$('#wp-submit').on('click', function(e){secupressOTPHandler(e);});
		$('#user_login').on('blur', function(e) {
			if ( ! $(this).val().length ) {
				return;
			}
			if ($('#user_pass:visible').length) {
				secupressOTPHandler(e);
				$('#secupress-areyouhuman').show();
			} else {
				$('#wp-submit').trigger('click');
				$('#user_pass:visible').trigger('focus');
				setTimeout(function() {
					if ( $('#user_pass:visible').length ) {
						$('#secupress-areyouhuman').show();
					}
				}, 250);
			}
		});
		if (typeof localStorage !== 'undefined') {
			var secupress_LS_login_data = localStorage.getItem('secupress_LS_login_data');
			if (secupress_LS_login_data) {
				var secupress_LS_login_data = JSON.parse(secupress_LS_login_data);
				if ( ! secupress_LS_login_data.login ) {
					localStorage.removeItem('secupress_LS_login_data');	
				} else {
					$('#user_login').val( secupress_LS_login_data.login );
					$('.forgetmenot').hide();
				}
				$('#wp-submit').trigger('click');
			}
		}
	});
	</script>
	<?php
	ob_end_flush();
}

add_action( 'wp_ajax_nopriv_otp_status', 'secupress_otpauth_otp_status_admin_ajax_cb' );
add_action( 'wp_ajax_otp_status', 'secupress_otpauth_otp_status_admin_ajax_cb' );
/**
 * Get qrcode status
 *
 * @since 2.2.6
 * @author Julio Potier
 **/
function secupress_otpauth_otp_status_admin_ajax_cb() {
	if ( ! isset( $_GET['l'] ) ) {
		wp_send_json_error( 0 );
	}
	$user = get_user_by( 'login', $_GET['l'] );
	if ( ! user_can( $user, 'exist' ) ) {
		$user = get_user_by( 'email', $_GET['l'] );
	}
	if ( ! user_can( $user, 'exist' ) ) {
		wp_send_json_error( 0 );
	}
	if ( secupress_is_user( $user ) ) {
		// success = need OTP
		if ( secupress_is_affected_role( 'users-login', 'double-auth', $user ) &&
			secupress_otpauth_get_user_option( 'secret', $user->ID ) &&
			secupress_otpauth_get_user_option( 'verified', $user->ID )
		) {
			wp_send_json_success();
		} else{
			wp_send_json_error( [ 'gravatar' => md5( $user->user_email ), 'display_name' => $user->display_name ] );
		}
	}
	wp_send_json_error( 0 );
}

/**
 * Add avatar CSS on demand
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_print_head_css( $user ) {
	?>
	<style type="text/css">
	.login h1 a{background-image: url('http://<?php echo rand( 0, 2 ); ?>.gravatar.com/avatar/<?php echo md5( $user->user_email ); ?>?s=180&d=<?php echo admin_url( '/images/wordpress-logo.svg?ver=20131107' ); ?>') !important; border-radius: 100%}
	.wrong{color:red}
	</style>
	<?php
}

add_filter( 'authenticate', 'secupress_otpauth_login_authenticate', 200, 3 );
/**
 * Try to log the user using a OTP code
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (object) $raw_user A WP_Error object if user is not correctly authenticated, a WP_User object if he is.
 * @param (string) $username A username or email address.
 * @param (string) $password User password.
 *
 * @return (object) A WP_Error or WP_User object.
 */
function secupress_otpauth_login_authenticate( 
	$raw_user, 
	$username, 
	#[\SensitiveParameter] 
	$password
) {
	static $running     = false;
	$incorrect_password = false;
	// Errors from other plugins.
	if ( is_wp_error( $raw_user ) ) {
		// Remove WP errors related to empty fields.
		unset( $raw_user->errors['empty_username'], $raw_user->errors['empty_password'] );

		if ( isset( $raw_user->errors['incorrect_password'] ) ) {
			$incorrect_password = $raw_user->errors['incorrect_password'];
			unset( $raw_user->errors['incorrect_password'] );
		}
		if ( $raw_user->errors ) {
			// There are still errors, don't go further.
			$running = false;
			return $raw_user;
		}
	}
	if ( $running || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
		return $raw_user;
	}
	$running = true;

	// Make sure to process only credentials provided by the login form.
	$username = ! empty( $_POST['log'] ) && $username === $_POST['log'] ? $username : ''; // WPCS: CSRF ok.
	$password = ! empty( $_POST['pwd'] ) && $password === $_POST['pwd'] ? $password : ''; // WPCS: CSRF ok.
	$user_id  = isset( $raw_user->ID ) ? $raw_user->ID : 0;

	if ( ! $username && ! $password || $username && $password ) {
		$running = false;
		if ( $incorrect_password &&
			secupress_is_affected_role( 'users-login', 'double-auth', $raw_user ) &&
			secupress_otpauth_get_user_option( 'secret', $user_id ) &&
			secupress_otpauth_get_user_option( 'verified', $user_id )
		) {
			$error_msg = __( 'You cannot login using your password, please just fill your login or email address.', 'secupress' );
			$wp_errors = new WP_Error( 'authentication_failed', $error_msg );
		} else {
			$button    = '— ' . __( 'or', 'secupress' ) . ' <button id="magic-link" name="magic_link" value="1" class="button-small button-link">' . __( 'Log In using a Magic Link', 'secupress' ) . '</button>';
			$wp_errors = new WP_Error( 'authentication_failed', $button, 'message' );
		}
		return $wp_errors;
	}
	if ( $username && isset( $_POST['need_otpauth'] ) ) {
		if ( ! secupress_is_user( $raw_user ) ) {
			$by           = is_email( $username ) ? 'email' : 'login';
			$temp_user    = get_user_by( $by, $username );
			// if the login was a fake email like "foobar@example.com", try now with a login.
			if ( ! secupress_is_user( $temp_user ) && is_email( $username ) ) {
				$raw_user = get_user_by( 'login', $username );
			} else {
				$raw_user = $temp_user;
			}
		}
	}
	$user_id  = isset( $raw_user->ID ) ? $raw_user->ID : 0;

	if ( $user_id > 0 && secupress_is_user( $raw_user ) && ! empty( $_POST ) &&
		secupress_is_affected_role( 'users-login', 'double-auth', $raw_user ) &&
		secupress_otpauth_get_user_option( 'secret', $user_id ) &&
		secupress_otpauth_get_user_option( 'verified', $user_id )
	) {

		$token       = wp_hash( wp_generate_password( 32, false ) );

		secupress_otpauth_update_user_option( 'token', $token, $user_id );
		secupress_otpauth_update_user_option( 'timeout', time() + 10 * MINUTE_IN_SECONDS, $user_id );

		$wp_lang     = get_user_locale( $user_id );
		$raw_user    = null;
		$rememberme  = (int) isset( $_POST['rememberme'] );
		$redirect_to = add_query_arg( array(
			'action'     => 'otp_needed',
			'token'      => $token,
			'rememberme' => $rememberme,
			'wp_lang'    => $wp_lang,
		), wp_login_url( '', true ) );

		wp_safe_redirect( $redirect_to );
		die();
	}
	// $rememberme  = (int) isset( $_POST['rememberme'] );
	// $redirect_to = add_query_arg( array(
	// 	'action'     => 'otp_not_needed',
	// 	'rememberme' => $rememberme,
	// ), wp_login_url( '', true ) );

	// wp_safe_redirect( $redirect_to );
	// die();
}

add_action( 'login_form_otp_needed', 'secupress_otpauth_needed_login_form_add_form' );
/**
 * The user need OTP now
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_needed_login_form_add_form() {
	global $wpdb;

	$errors              = null;
	$message             = '';
	$do_delete           = false;
	$show_form           = true;
	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['rememberme'] = isset( $_REQUEST['rememberme'] );
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$CLEAN['otp']        = isset( $_POST['otp'] ) ? preg_replace('/[^a-fA-F0-9]/', '', $_POST['otp'] ) : '';
	$CLEAN['otp']        = isset( $_POST['otp'] ) && ( $CLEAN['otp'] === $_POST['otp'] ) ? $CLEAN['otp'] : '';
	$user                = get_user_by( 'id', $CLEAN['uid'] );
	$time                = secupress_otpauth_get_user_option( 'timeout', $CLEAN['uid'] );

	if ( $time < time() ) {
		/**
		 * Run an action on OTP login error
		 * 
		 * @param (WP_User) $user
		 * @param (string)  $context
		 */
		do_action( 'secupress.plugins.otp-auth.login.error', $user, 'expired_time' );

		$show_form       = false;
		$do_delete       = true;
		$message         = sprintf( __( 'Too much time has elapsed between the first step and now.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ) . '</p>', wp_login_url( '', true ) );

	} elseif ( ! $CLEAN['token'] || ! $user ) {

		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );

	} elseif ( $CLEAN['otp'] && isset( $_POST['token'] ) ) {

		$do_login = false;
		if ( 10 === mb_strlen( $CLEAN['otp'] ) ) {
			// Backup code attempt
			$backup_codes   = secupress_otpauth_get_user_option( 'backupcodes', $user->ID );
			if ( ! $backup_codes ) {
				die('You ran out f backup codes, ask a code in your email');
			}
			$key      = array_search( $CLEAN['otp'], $backup_codes );
			/**
			 * Let the possibility to bypass/force the OTP backup code verification
			 * 
			 * @param (string|bool) $key The key index found in the array, or false
			 * @param (string)      $otp The user code given in the form
			 */
			$key      = apply_filters( 'secupress.plugins.otp-auth.backupcode.verify', $key, $CLEAN['otp'] );
			if ( $key === false ) {
				/**
				 * @see Scroll up :)
				 */
				do_action( 'secupress.plugins.otp-auth.login.error', $user, 'bad_backup_code' );

				$show_form  = true;
				$do_delete  = false;
				$message    = __( '<strong>Error</strong>: Wrong Authentication Code.', 'secupress' );
				if ( function_exists( 'secupress_limitloginattempts_add_one_try' ) ) {
					secupress_limitloginattempts_add_one_try( $user->ID );
				}
				add_action( 'login_footer', 'wp_shake_js', 12 );
			} else {
				unset( $backup_codes[ $key ] );
				secupress_otpauth_update_user_option( 'backupcodes', $backup_codes, $user->ID );
				$do_login = true;
			}

		} else {
			// OTP code attempt
			$otpauth_secret = secupress_otpauth_get_user_option( 'secret', $CLEAN['uid'] );
			$lasttimeslot   = secupress_otpauth_get_user_option( 'lasttimeslot', $CLEAN['uid'] );

			if ( $timeslot  = secupress_base32_verify( $otpauth_secret, $CLEAN['otp'], $lasttimeslot ) ) {
				secupress_otpauth_update_user_option( 'lasttimeslot', $timeslot, $user->ID );
				$do_login = true;
			} else {
				/**
				 * @see Scroll up :)
				 */
				do_action( 'secupress.plugins.otp-auth.login.error', $user, 'invalid_otp_code' );

				$do_delete = false;
				$show_form = true;
				$message   = __( '<strong>Error</strong>: Wrong Authentication Code.', 'secupress' );
				if ( function_exists( 'secupress_limitloginattempts_add_one_try' ) ) {
					secupress_limitloginattempts_add_one_try( $user->ID );
				}

				add_action( 'login_footer', 'wp_shake_js', 12 );
			}
		}

		if ( $do_login ) {
			$args = array( 'user_login' => $user->user_login, 'user_password' => time() );
			/**
			 * @see WP /wp-includes/user.php
			 */ 
			$secure_cookie = apply_filters( 'secure_signon_cookie', secupress_server_is_ssl(), $args ); // we don't have the real password, just pass something.

			wp_set_auth_cookie( $CLEAN['uid'], $CLEAN['rememberme'], $secure_cookie );

			secupress_otpauth_delete_user_option( 'token', $CLEAN['uid'] );
			secupress_otpauth_delete_user_option( 'timeout', $CLEAN['uid'] );
			secupress_otpauth_delete_user_option( 'seed', $CLEAN['uid'] );
			secupress_otpauth_delete_user_option( 'qrcode-status', $CLEAN['uid'] );

			/**
			 * @see WP /login.php
			 */ 
			do_action( 'wp_login', $user->user_login, $user );

			$redirect_to = apply_filters( 'login_redirect', admin_url(), admin_url(), $user );

			/**
			 * Run an action on OTP auth login success before redirection
			 * 
			 * @param (WP_User) $user
			 */
			do_action( 'secupress.plugins.otp-auth.login.success', $user );

			wp_redirect( $redirect_to );
			die();
		}
	}

	if ( $do_delete ) {
		secupress_otpauth_delete_user_option( 'token', $CLEAN['uid'] );
		secupress_otpauth_delete_user_option( 'timeout', $CLEAN['uid'] );
	}

	$mobile_sess = secupress_user_has_mobile_session( $CLEAN['uid'] );

	$title   = $message ? __( 'Error', 'secupress' ) : __( 'Log In', 'secupress' );
	$message = $message ? '<div class="notice notice-error">' . $message . '</div>' : '';
	login_header( $title, $message );

	if ( $show_form ) {

		$attempts = isset( $_POST['attempts'] ) ? (int) $_POST['attempts'] : 0;
		++$attempts;

		secupress_otpauth_print_head_css( $user );
		?>
		<form name="loginform" id="loginform" action="<?php echo esc_url( site_url( 'wp-login.php?action=otp_needed' ) ); ?>" method="post">
			<p>
				<p class="input"><?php printf( __( 'Howdy, %s' ), '<span class="display-name">' . $user->display_name . '</span>' ); ?></p>
				<span>
					<button id="fake-input-switch" type="button" class="button-small button-link">
						<?php _e( 'Not you? Switch account.', 'secupress' ); ?>
					</button>
				</span>
				<hr>
				<label>
					<h3><?php printf( __( '2-Step Verification for %s', 'secupress' ), get_bloginfo( 'name', 'display' ) ); ?></h3>
					<span class="authinfo">
						<img src="<?php echo plugins_url( '/inc/img/otp-mobile.png', __FILE__ ); ?>" alt="" role="presentation">
						<em id="enter_code"><?php _e( 'Enter the verification code generated by your mobile application.', 'secupress' ); ?></em>
						<em id="enter_backup" class="hidden"><?php _e( 'Enter one of your backup codes to log-in.', 'secupress' ); ?></em>
					</span>
					<input type="text" placeholder="******" class="otpauth" onkeypress="return event.charCode >= 48 && event.charCode <= 57 || event.charCode == 13" name="otp" id="otp" maxlength="6" size="20" required pattern="[0-9]{6}" style="ime-mode: inactive;" />
				</label>
			</p>

			<input type="hidden" name="token" value="<?php echo esc_attr( $CLEAN['token'] ); ?>">
			<?php if ( $CLEAN['rememberme'] ) { ?>
				<input type="hidden" name="rememberme" value="forever">
			<?php } ?>
			<input type="hidden" name="attempts" value="<?php echo $attempts; ?>">

			<p class="submit">
				<input type="submit" name="wp-submit" id="main-submit" class="button button-primary button-large button-full" value="<?php echo esc_attr_x( 'Verify', 'verb', 'secupress' ); ?>" />
			</p>
		</form>

		<form action="<?php echo esc_url( site_url( 'wp-login.php?action=otpauth_lost_redir' ) ); ?>" method="post" id="form_alt">
			<input type="hidden" name="token" value="<?php echo esc_attr( $CLEAN['token'] ); ?>">
			<?php if ( $CLEAN['rememberme'] ) { ?>
				<input type="hidden" name="rememberme" value="forever">
			<?php } ?>

			<?php 
			$hide_or_not = $attempts <= 1 ? '' : ' hidden';
			?>
			<button type="button" class="no-button button-full<?php echo $hide_or_not; ?>" id="help_lost"><?php _e( 'Having trouble with your code?', 'secupress' ); ?> <span class="dashicons dashicons-arrow-right-alt2 dashright"></span></button>
			<?php 
			$hide_or_not = $attempts > 1 ? '' : 'hide-if-js';
			?>
			<div id="help_info" class="<?php echo $hide_or_not; ?>">
				<h3><?php _e( 'Try one of these alternate methods:', 'secupress' ); ?></h3>
				<ul>
					<li>
						<button class="button-small button-link" type="button" id="backup_use">
							<?php _e( 'Enter a backup code to proceed', 'secupress' ); ?>
						</button>
					</li>
					<li>
						<button class="button-small button-link" id="backup_send" value="backup_send" name="otpauth_lost_method">
							<?php _e( 'Send a backup code to my email', 'secupress' ); ?>
						</button>
					</li>
					<?php 
					if ( $mobile_sess && secupress_get_module_option( 'double-auth_otp-mobile', 1, 'users-login' ) && ! secupress_otpauth_get_user_option( 'dont-otp-mobile', $user->ID ) ) {
					?>
					<li>
						<button class="button-small button-link" id="mobile_session" value="mobile_session" name="otpauth_lost_method">
							<?php _e( 'Log in using your mobile phone by scanning a QR code', 'secupress' ); ?>
						</button>
					</li>
				<?php } ?>
				</ul>
			</div>
		</form>
		<?php
		login_footer( 'otp' );
	} else {
		login_footer();
	}
	die();
}

add_action( 'login_form_otpauth_lost_redir', 'secupress_otpauth_lost_form_redir' );
/**
 * User has lost the way to log in, help them
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_lost_form_redir() {
	global $wpdb;
	if ( ! isset( $_POST['otpauth_lost_method'] ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['rememberme'] = isset( $_REQUEST['rememberme'] );
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = get_user_by( 'id', $CLEAN['uid'] );

	if ( ! $CLEAN['token'] || ! user_can( $user, 'exist' ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	$time = secupress_otpauth_get_user_option( 'timeout', $CLEAN['uid'] );

	if ( $time >= time() ) {

		switch ( $_POST['otpauth_lost_method'] ) {
			case 'backup_send':
				$codes = secupress_otpauth_get_user_option( 'backupcodes', $user->ID );

				if ( is_array( $codes ) && count( array_filter( $codes ) ) ) {

		 			$codes = array_filter( $codes );
		 			$code  = reset( $codes );

				} else {

					$redirect_to = add_query_arg( array(
						'action'     => 'otpauth_no_backup_codes',
						'token'      => $CLEAN['token'],
					), wp_login_url( '', true ) );

					wp_redirect( $redirect_to );
					die();

				}

				$subject = sprintf( __( '[%s] OTP Authenticator Backup Code request', 'secupress' ), '###SITENAME###' );
				/**
				 * Filter the subject from the mail that gives the user a backup code
				 * 
				 * @param (string) $subject
				 * @param (WP_User) $user
				 */
				$subject = apply_filters( 'secupress.plugins.otp-auth.backupcode_email.subject', $subject, $user );

				$otpauth_backupcodes = secupress_otpauth_get_user_option( 'backupcodes', $user->ID );				
				$backup_codes_count  = count( array_filter( (array) $otpauth_backupcodes ) ) - 1; // we substract this one.
				$message             = sprintf( __( 'Hello %1$s, you asked for a backup code, here it comes: %2$s. You have %3$d backup codes left.' ), $user->display_name, secupress_code_me( $code ), $backup_codes_count );

				/**
				 * Filter the message from the mail that gives the user a backup code
				 * 
				 * @param (string) $subject
				 * @param (WP_User) $user
				 */
				$message = apply_filters( 'secupress.plugins.otp-auth.backupcode_email.message', $message, $user );

				secupress_send_mail( $user->user_email, $subject, $message );

				$redirect_to = add_query_arg( array(
					'action'     => 'otpauth_lost_form_backupcode',
					'token'      => $CLEAN['token'],
					'rememberme' => $CLEAN['rememberme'],
					'emailed'    => 1,
				), wp_login_url( '', true ) );

				wp_redirect( $redirect_to );
				die();
			break;
			
			case 'mobile_session':
				if ( secupress_get_module_option( 'double-auth_otp-mobile', 1, 'users-login' ) && ! secupress_otpauth_get_user_option( 'dont-otp-mobile', $user->ID ) ) {
					$redirect_to = add_query_arg( array(
						'action'     => 'otpauth_need_mobile',
						'token'      => $CLEAN['token'],
						'rememberme' => $CLEAN['rememberme'],
					), wp_login_url( '', true ) );

					wp_redirect( $redirect_to );
					die();
				}			
			default:
				secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
			break;
		}

	} else {

		/**
		 * @see Scroll up :)
		 */
		do_action( 'secupress.plugins.otp-auth.login.error', $user, 'expired_time' );

		$message = sprintf( __( 'Too much time has elapsed between the first step and now.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ) . '</p>', wp_login_url( '', true ) );
		login_header( __( 'Error', 'secupress' ), '<p class="notice notice-error">' . $message . '</p>' );
		login_footer();
		die();
	}
}

add_action( 'login_form_otpauth_need_mobile', 'secupress_passwordless_otpauth_need_mobile' );
/**
 * Modify the login header page message for our action
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_passwordless_otpauth_need_mobile() {

	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['rememberme'] = isset( $_REQUEST['rememberme'] );
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = get_user_by( 'id', $CLEAN['uid'] );
	$suffix              = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	if ( ! $CLEAN['token'] || ! user_can( $user, 'exist' ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	$time = secupress_otpauth_get_user_option( 'timeout', $CLEAN['uid'] );

	if ( $time < time() ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}
	secupress_otpauth_print_head_css( $user );
	$seed      = secupress_generate_key( 32 );
	secupress_otpauth_update_user_option( 'seed', $seed, $CLEAN['uid'] );
	secupress_otpauth_delete_user_option( 'qrcode-status', $CLEAN['uid'] );
	$token_url = add_query_arg( [ 'action' => 'otpauth_mobile_auth', 'token' => $CLEAN['token'], 'seed' => $seed ], wp_login_url( '', true ) );
	$message   = '<h2>' . sprintf( __( 'Howdy, %s' ), '<span class="display-name">' . esc_html( $user->display_name ) . '</span>' ) . '</h2>'; // do not translate "howdy", ppl may have hacked this already from WP domain
	login_header( __( 'Log In', 'secupress' ), '<div class="message">' . $message . '</div>' );
	?>
	<style type="text/css">
		#otpauth_qrcode{image-rendering: pixelated;transition: filter 0.6s ease;}
	</style>
	<div class="notice notice-error hide-if-js"><?php _e( '<strong>Error</strong>.<br>This module needs JavaScript activated.', 'secupress' ); ?></div>
	<div class="notice notice-info         notice-response" id="notice-basic"   ><?php _e( 'Please scan the <abbr title="Quick Response Code">QR Code</abbr> below with your mobile phone that is already logged in.', 'secupress' ); ?></div>
	<div class="notice notice-info  hidden notice-response" id="notice-pending" ><?php _e( '<strong>Reviewing request</strong>.<br>Awaiting owner‘s decision.', 'secupress' ); ?> <img src="<?php echo admin_url( '/images/wpspin_light.gif' ); ?>" /></div>
	<div class="notice success      hidden notice-response" id="notice-accepted"><?php _e( '<strong>Request accepted</strong>.<br>Redirection in 3&hellip; 2&hellip; 1&hellip;', 'secupress' ); ?></div>
	<div class="notice notice-error hidden notice-response" id="notice-refused" ><?php _e( '<strong>Request refused</strong>.<br>Sorry, the request has been refused by the session owner.', 'secupress' ); ?><br><?php wp_loginout(); ?></div>
	<div class="notice notice-error hidden notice-response" id="notice-expired" ><?php _e( '<strong>Request expired</strong>.<br>Too much time has elapsed between the first step and now.', 'secupress' ); ?><br><?php wp_loginout(); ?></div>
	<form id="qrcode">
		<span id="otpauth_qrcode"></span>
	</form>
	<?php 
	wp_enqueue_script( 'jquery' );
	wp_enqueue_script( 'secupress-jquery-qrcode' );
	wp_enqueue_script( 'secupress-qrcode' );
	?>
	<script type="text/javascript">
	document.addEventListener("DOMContentLoaded", function() {

		jQuery(document).ready(function($) {
			let qrcode = "<?php echo $token_url; ?>";
			$( "#otpauth_qrcode" ).text( "" ).qrcode( { "text": qrcode, "width": 270, "height": 270 } );

			$('.hide-if-js, .hidden').hide();
			var ajaxurl    = '<?php echo esc_js( esc_url( admin_url( 'admin-ajax.php' ) ) ); ?>';
			var intervalID = setInterval(function() {
				var params = {
					nonce: '<?php echo esc_js( wp_create_nonce( 'otp-qrcode-status' ) ); ?>',
					action: 'otp_qrcode_status',
					token: '<?php echo esc_js( $CLEAN['token'] ); ?>'
				};
				$.ajax({
					url: ajaxurl,
					type: 'POST',
					data: params,
					dataType: 'json',
					success: function(response) {
						if ( '0' !== response && ( response.data || ! response.success ) ) {
							$('.notice').hide();
							$('#otpauth_qrcode').parent().slideUp('slow', function(){$(this).remove();});
						}
						if (response.success === true && response.data === 'pending') {
							$('#notice-pending').show();
						} else if (response.success === true && response.data && response.data !== 'pending') {
							clearInterval(intervalID);
							$('#notice-accepted').show();
							setTimeout(function() {
								window.location.href = response.data;
							}, 2000);
						} else if (response.success === false && response.data === 'rejected' ) {
							clearInterval(intervalID);
							$('#notice-refused').show();
						} else if (response.success === false && response.data === 'expired' ) {
							clearInterval(intervalID);
							$('#notice-expired').show();
						}
					},
					error: function() {
						console.error('Error fetching data from server.');
						clearInterval(intervalID);
					}
				});
			}, 2000);
		});

	});
	</script>
	<?php
	login_footer();
	die();
}

add_action( 'wp_ajax_nopriv_otp_qrcode_status', 'secupress_otpauth_qrcode_status_admin_ajax_cb' );
/**
 * Run an ajax call every 2 sec to get the qrcode status
 *
 * @author Julio Potier
 * @since 2.2.6
 **/
function secupress_otpauth_qrcode_status_admin_ajax_cb() {
	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	// $CLEAN['rememberme'] = isset( $_REQUEST['rememberme'] );
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = get_user_by( 'id', $CLEAN['uid'] );

	if ( ! $CLEAN['token'] || ! user_can( $user, 'exist' ) ) {
		wp_send_json_error();
	}

	$time = secupress_otpauth_get_user_option( 'timeout', $CLEAN['uid'] );

	if ( $time < time() ) {
		secupress_otpauth_delete_user_option( 'seed', $CLEAN['uid'] );
		secupress_otpauth_delete_user_option( 'qrcode-status', $CLEAN['uid'] );
		wp_send_json_error('expired');// DO NOT TRANSLATE
	}

	$status = secupress_otpauth_get_user_option( 'qrcode-status', $CLEAN['uid'] );
	switch ( $status ) {
		case 'pending':
			wp_send_json_success( $status );
		break;

		case strpos( 'http', $status ) === 0:
			secupress_otpauth_delete_user_option( 'seed', $CLEAN['uid'] );
			secupress_otpauth_delete_user_option( 'qrcode-status', $CLEAN['uid'] );
			wp_send_json_success( $status );
		break;

		case 'rejected':
			wp_send_json_error('rejected'); // DO NOT TRANSLATE
		break;
	}
	wp_send_json_success( $status );
}

add_action( 'login_form_otpauth_no_backup_codes', 'secupress_passwordless_otpauth_no_backup_codes' );
/**
 * Modify the login header page message for our action
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_passwordless_otpauth_no_backup_codes() {
	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = get_user_by( 'id', $CLEAN['uid'] );

	if ( ! $CLEAN['token'] || ! user_can( $user, 'exist' ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}
	login_header( __( 'Error', 'secupress' ), '<p class="notice notice-error">' . __( 'You have exhausted your backup codes despite our warnings, and you are currently unable to log in.<br>Please contact an administrator for assistance.', 'secupress' ) . '</p>' );
	login_footer();
	die();
}

add_action( 'login_form_otpauth_lost_form_backupcode', 'secupress_passwordless_otpauth_lost_form_backupcode' );
/**
 * Modify the login header page message for our action
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_passwordless_otpauth_lost_form_backupcode() {
	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = get_user_by( 'id', $CLEAN['uid'] );

	if ( ! $CLEAN['token'] || ! user_can( $user, 'exist' ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	$message = sprintf( __( 'Please check your email for the Backup Code needed to <a href="%s">log in</a>.', 'secupress' ), wp_login_url( '', true ) );
	login_header( __( 'Log In', 'secupress' ), '<p class="message">' . $message. '</p>' );
	login_footer();
	die();
}

add_action( 'login_form_otpauth_mobile_auth', 'secupress_passwordless_otpauth_mobile_auth' );
/**
 * Modify the login header page message for our action
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_passwordless_otpauth_mobile_auth() {
	if ( ! secupress_is_mobile() ) {
		login_header( __( 'Error', 'secupress' ), '<div class="notice notice-error">' . __( 'This page is not accessible on a desktop. Please use your mobile device instead.', 'secupress' ) . '</div>' );
		login_footer();
		die();
	}
	if ( ! is_user_logged_in() ) {
		login_header( __( 'Error', 'secupress' ), '<div class="notice notice-error">' . sprintf( __( 'You need to <a href="%s">log in</a> first to access this page.', 'secupress' ), wp_login_url( '', true ) ) . '</div>' );
		login_footer();
		die();
	}
	$CLEAN               = array();
	$CLEAN['token']      = isset( $_REQUEST['token'] ) ? sanitize_key( $_REQUEST['token'] ) : false;
	$CLEAN['seed']       = isset( $_REQUEST['seed'] ) ? sanitize_key( $_REQUEST['seed'] ) : '';
	$CLEAN['rememberme'] = isset( $_REQUEST['rememberme'] );
	$CLEAN['uid']        = secupress_otpauth_get_user_from_meta_value( 'token', $CLEAN['token'] );
	$CLEAN['uid']        = (int) reset( $CLEAN['uid'] );
	$user                = wp_get_current_user();
	$user_seed           = secupress_otpauth_get_user_option( 'seed', $user->ID ) ?? '';
	$title               = __( 'Double Authentication Account Settings', 'secupress' );

	if ( ! $CLEAN['token'] || ! $CLEAN['seed'] || ! $user_seed || ! user_can( $user, 'exist' ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	if ( 'pending' !== $user_seed && ! hash_equals( strtoupper( $CLEAN['seed'] ), $user_seed ) ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	$time = secupress_otpauth_get_user_option( 'timeout', $user->ID );

	if ( $time < time() ) {
		secupress_die( sprintf( __( 'Invalid Link.<br>Please try to <a href="%s">log in again</a>.', 'secupress' ), wp_login_url( '', true ) ), '', [ 'force_die' => true, 'context' => 'otpauth', 'attack_type' => 'login' ] );
	}

	// update the status to "pending", the desktop user will see a "thinking..." animation
	secupress_otpauth_update_user_option( 'qrcode-status', 'pending' ); // DO NOT TRANSLATE
	
	if ( isset( $_POST['otpauth_accept'] ) ) {
		if ( '1' === $_POST['otpauth_accept'] ) {
			$passwordless_url = secupress_passwordless_send_link( $user, [ 'sendmail' => false, 'rememberme' => $CLEAN['rememberme'] ] )['token_url'];
			secupress_otpauth_update_user_option( 'qrcode-status', $passwordless_url );
			secupress_action_page( $title, '<form class="wrap">' . __( 'Login request <strong>accepted</strong>. You may close this window.', 'secupress' ) . '</form>' );
		} else {
			secupress_otpauth_update_user_option( 'qrcode-status', 'rejected' ); // DO NOT TRANSLATE
			secupress_action_page( $title, '<form class="wrap">' . __( 'Login request <strong>rejected</strong>. You may close this window.', 'secupress' ) . '</form>' );
		}
	}
	$suffix        = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	$args['head']  = '<link href="' . esc_url( SECUPRESS_ADMIN_CSS_URL . 'secupress-wordpress' . $suffix . '.css' ) . '" media="all" rel="stylesheet">';
	$args['head'] .= '<script src="' . esc_url( site_url( '/wp-includes/js/jquery/jquery.js?ver=' . wp_get_wp_version() ) ) . '" type="text/javascript"></script>';
	$args['body']  = 'class="wp-core-ui"'; // Needed for the buttons CSS

	ob_start();
	?>
	<form method="post" class="wrap" action="" method="post">
		<h2><?php _e( 'You requested to log in on desktop using your mobile session.', 'secupress' ); ?></h2>
		<h3><?php _e( 'Do you accept the request?', 'secupress' ); ?></h3>
		<div>
			<button name="otpauth_accept" value="0" class="secupress-button secupress-button-secondary"><?php _e( 'Cancel the request', 'secupress' ); ?></button>
			<button name="otpauth_accept" value="1" class="secupress-button secupress-button-primary"><?php _e( 'Accept the request', 'secupress' ); ?></button>
		</div>
	</form>
	<?php
	$content = ob_get_contents();
	ob_clean();

	// Display the correct informations
	secupress_action_page( $title, $content, $args );
}

/**
 * Get a user_id from a meta value, only for otp auth
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (string) $key
 * @param (string) $value
 *
 * @return (array)
 */
function secupress_otpauth_get_user_from_meta_value( $key, $value ) {
	global $wpdb;
	$res = secupress_cache_data( "$key|$value" );
	if ( $res ) {
		return $res;
	}
	$res = $wpdb->get_col( $wpdb->prepare( "SELECT user_id FROM $wpdb->usermeta WHERE meta_key = %s AND meta_value = %s", secupress_otpauth_get_option_name( $key ), $value ) );
	secupress_cache_data( "$key|$value", $res );
	return $res;
}

add_action( 'login_footer', 'secupress_otpauth_footer_js' );
/**
 * Modify the UI of main login form
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_footer_js() {
	global $action;
	?>
	<script src="<?php echo esc_url( site_url( '/wp-includes/js/jquery/jquery.js' ) ); ?>" type="text/javascript"></script>
	<?php
	if ( 'otp_needed' === $action ) {
	?>
	<script type="text/javascript">
	jQuery( document ).ready( function($) {
		$( ".hidden, .hide-if-js" ).hide();
		$( "#help_lost" ).on( "click", function(e) {
			e.preventDefault();
			$( this ).hide();
			$( "#help_info" ).slideDown();
		} );
	} );
	</script>
	<?php
	}
	if ( 'login' === $action ) {
	?>
	<?php
	}
	?>
	<script type="text/javascript">
	jQuery( document ).ready( function($) {
		$('#secupress-areyouhuman').hide();
		$('#otp').on('keyup', function(e){
			if ( $(this).val().length === $(this).attr('placeholder').length ) {
				$('#loginform').trigger('submit');
			}
		});
		$('#backup_use').on('click', function(e){
			$('#otp').attr('placeholder', '**********')
			.attr('pattern', '[0-9A-F]{10}')
			.attr('onkeypress', 'return ( event.charCode >= 48 && event.charCode <= 57 ) || ( event.charCode >= 65 && event.charCode <= 70 ) || event.charCode == 13')
			.attr('maxlength', '10').val('').trigger('focus');
			$('div.notice').remove();
			$('#enter_code').hide();
			$('#enter_backup').show();
			$(this).parent().toggle('slow');
			var element = $('#form_alt');
			if (element.length) {
				element[0].scrollIntoView({ behavior: 'smooth' });
			}
		});
	} );
	</script>
	<?php
}

// add_filter( 'wp_login_errors', 'secupress_otpauth_login_errors' );
/**
 * Filter the WP errors to replace the "lost password" link by our magic link (formely PasswordLess)
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param  (WP_Error) $errors
 * @return (WP_Error) $errors
 */
function secupress_otpauth_login_errors( $errors ) {
	if ( ! isset( $_POST['log'] ) ) {
		return $errors;
	}

	$user = secupress_get_user_by( $_POST['log'] );
	if ( ! secupress_is_user( $user ) ) {
		return $errors;
	}

	foreach( $errors as &$error ) {
		foreach( $error as &$_error ) {
			if ( 'incorrect_password' === key( $error ) || 'incorrect_username' === key( $error ) ) {
				$button = '<button id="magic-link" name="magic_link" value="1" class="button-small button-link">' . __( 'Get my Magic Link', 'secupress' ) . '</button>';
				$a_link = '<a href="' . wp_lostpassword_url() . '">' . __( 'Lost your password?' ) . '</a>'; 
				$_error = str_replace( $a_link, $button, $_error );
			}
		}
	}
	return $errors;
}

add_action( 'edit_user_profile', 'secupress_otpauth_profile_user_options', 0 );
/**
 * Extend user profile page with double authentication settings.
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (WP_User) $user
 */
function secupress_otpauth_profile_user_options( $user ) {
	if ( ! current_user_can( 'edit_users' ) ) {
		return;
	}
	if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		return;
	}
	$user_id               = $user->ID;
	$otpauth_secret        = secupress_otpauth_get_user_option( 'secret', $user_id );
	$user_verified         = secupress_otpauth_get_user_option( 'verified', $user_id );
	$otpauth_backupcodes   = secupress_otpauth_get_user_option( 'backupcodes', $user_id );
	?>
	<h3 id="otpauth_secret"><?php _e( 'OTP Authenticator Settings', 'secupress' ); ?></h3>
	<table class="form-table">
		<tr>
			<th><?php _e('Secret OTP Key', 'secupress'); ?></th>
			<td>
				<?php
				if ( $user_verified ) {
					echo '<p>' . __( 'Their account is <strong>protected</strong> by a double authentication.', 'secupress' ) . '</p>';
				} else {
					echo '<p>' . __( 'Their account is <strong>not protected</strong> by a double authentication.', 'secupress' ) . '</p>';
				}
				?>
				<?php
				if ( $user_verified ) {
					?><a href="<?php echo wp_nonce_url( admin_url( 'admin-post.php?action=secupress_otpauth_regen_secret&user=' . $user_id ), 'secupress_otpauth_regen_secret-' . $user_id ); ?>" class="button button-secondary button-small"><?php
					_e( 'Revoke their application key', 'secupress' );
					?></a><?php
				}
				?>
			</td>
		</tr>
	</table>
	<?php
}

add_action( 'profile_personal_options', 'secupress_otpauth_profile_personal_options' );
/**
 * Extend personal profile page with double authentication settings.
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_profile_personal_options() {
	global $current_user;
	if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $current_user ) ) {
		return;
	}

	$user_id               = $current_user->ID;
	$otpauth_secret        = secupress_otpauth_get_user_option( 'secret' );
	$user_verified         = secupress_otpauth_get_user_option( 'verified' );
	$otpauth_backupcodes   = secupress_otpauth_get_user_option( 'backupcodes' );
							/* translators: 1 is a link to an Android app on google site, 2 is a link to an iOS app on apple site */
	$app_message           = __( 'Get a Free <abbr title="One-Time Password">OTP</abbr> Mobile Application on %1$s and %2$s to setup your OTP App.', 'secupress' );
	?>
	<h3 id="otpauth_secret"><?php _e( 'OTP Authenticator Settings', 'secupress' ); ?></h3>

	<table class="form-table">
		<tbody>
			<tr>
				<th>
					<?php _e( 'Secret OTP Key', 'secupress' ); ?>
					<p class="description"><?php
						if ( ! $user_verified ) {
							if ( apply_filters( 'secupress.plugins.otp-auth.show_app_links', true ) ) {
								printf(
									$app_message,
									'<a href="' . __( 'https://play.google.com/store/apps/details?id=org.fedorahosted.freeotp', 'secupress' ) . '" target="_blank">Android</a>',
									'<a href="' . __( 'https://itunes.apple.com/us/app/freeotp-authenticator/id872559395?mt=8', 'secupress' ) . '" target="_blank">iOS</a>'
								);
							} else {
								printf(
									$app_message,
									'Android',
									'iOS'
								);
							}
						}
					?></p>
				</th>
				<td>
					<?php
					if ( $user_verified ) {
						echo '<p>' . __( 'Your account is <strong>using</strong> a double authentication.', 'secupress' ) . '</p>';
					} else {
						echo '<p>' . __( 'Your account is <strong>not using</strong> a double authentication.', 'secupress' ) . '</p>';
					}
					?>
					<a href="<?php echo wp_nonce_url( admin_url( 'admin-post.php?action=secupress_otpauth_regen_secret' ), 'secupress_otpauth_regen_secret' ); ?>" class="button button-secondary button-small">
						<?php
						if ( $user_verified ) {
							_e( 'Revoke and generate a new application key', 'secupress' );
						} else {
							_e( 'Generate an application key', 'secupress' );
						}
						?>
					</a>
				</td>
			</tr>			
			<?php 
			// Display the setting "get alerted by email when OTP is used"
			if ( $user_verified && secupress_get_module_option( 'double-auth_otp-mail', 1, 'users-login' ) ) {
				$checked = checked( secupress_otpauth_get_user_option( 'dont-otp-mail' ), false, false );
			?>
			<tr>
				<th>
					<?php _e( 'Get alerted when the OTP Auth App is used', 'secupress' ); ?>
				</th>
				<td>
					<label><input type="checkbox" name="double-auth_otp-mail" value="1" <?php echo $checked; ?>> <?php _e( 'Yes, send me an email when someone logs into my account using OTP', 'secupress' ); ?></label>
					<input type="hidden" name="double-auth_otp-mail_flag" value="1">
				</td>
			</tr>
			<?php
			}
			// Display the setting "allow mobile session to be used"
			if ( $user_verified && secupress_get_module_option( 'double-auth_otp-mobile', 1, 'users-login' ) ) {
				$checked = checked( secupress_otpauth_get_user_option( 'dont-otp-mobile' ), false, false );
			?>
			<tr>
				<th>
					<?php _e( 'Allow mobile sessions to log in', 'secupress' ); ?>
				</th>
				<td>
					<label><input type="checkbox" name="double-auth_otp-mobile" value="1" <?php echo $checked; ?>> <?php _e( 'Yes, I want to be able to use my already logged in mobile session', 'secupress' ); ?></label>
					<input type="hidden" name="double-auth_otp-mobile_flag" value="1">
				</td>
			</tr>
			<?php
			}
			// Display backup codes if the auth is verified (so we have backup codes).
			if ( $user_verified ) {
				$backup_codes_count        = count( array_filter( (array) $otpauth_backupcodes ) );
				$backup_codes_count_string = sprintf( _n( 'You have %d unused code.', 'You have %d unused codes.', $backup_codes_count, 'secupress' ), $backup_codes_count );
				?>
				<tr>
					<th>
						<?php _e( 'Backup Codes', 'secupress' ); ?>
						<p class="description"><?php _e( 'When your smartphone is unavailable or just can\'t log in your account using the Mobile Authenticator.', 'secupress' ); ?></p>
					</th>
					<td>
						<p id="backupcodes_codes_description" data-desc="<?php echo esc_attr( $backup_codes_count_string ); ?>" class="description">
							<?php
							echo esc_html( $backup_codes_count_string );
							?>
						</p>

						<div id="backupcodes_codes" class="hide-if-js">
							<ol>
								<?php
								foreach ( $otpauth_backupcodes as $bkcode ) {
									echo secupress_tag_me( secupress_code_me( $bkcode ), 'li' );
								}
								?>
							</ol>
						</div>

						<p id="backupcodes_warning" class="hidden description">
							<?php _e( 'Keep them someplace accessible, like your wallet. Each code can be used only once.<br/>Before running out of backup codes, generate new ones. Only the latest set of backup codes will work.', 'secupress' ); ?>
						</p>

						<?php if ( $backup_codes_count > 0 ) { ?>
							<p id="backupcodes_show_button" class="hide-if-no-js">
								<button class="button button-secondary button-small" type="button">
									<?php _e( 'Show backup codes', 'secupress' ); ?>
								</button>
							</p>
						<?php } ?>

						<p>
							<a href="<?php echo wp_nonce_url( admin_url( 'admin-post.php?action=secupress_otpauth_renew_backup_codes' ), 'secupress_otpauth_renew_backup_codes' ); ?>" id="otpauth_newcodes" class="button button-secondary button-small">
								<?php _e( 'Generate new backup codes', 'secupress' ); ?>
							</a>
						</p>
					</td>
				</tr>
				<?php
			}
			?>
		</tbody>
	</table>
	<?php 
	if ( $user_verified ) {
	?>
	<script type="text/javascript">
	jQuery( document ).ready( function($) {
		// otpauth_new_backup_codes
		$( "#otpauth_newcodes" ).on( "click", function( e ) {
			e.preventDefault();

			if ( confirm( '<?php echo esc_js( __( "Renewing your backup codes will revoke all old ones.\nAre you sure to continue?", 'secupress' ) ); ?>' ) ) {
				let href = $( this ).attr( "href" );

				$( '#backupcodes_codes li' ).html( '<img src="<?php echo admin_url( '/images/wpspin_light.gif' ); ?>" alt="" role="presentation" />' );
				$( '#backupcodes_codes, #backupcodes_warning' ).show();
				$( '#backupcodes_show_button' ).hide();

				$.get( href.replace( "admin-post", "admin-ajax" ), function( data ) {
					var lis, orig, code;

					if ( data.success ) {
						$( '#backupcodes_codes_description' ).text( $( '#backupcodes_codes_description' ).data( 'desc' ) );

						lis = '<ol>';
						for ( index in data.data ) {
							code = data.data[ index ];
							lis += '<li><code>' + code + '</code></li>';
						}
						lis += '</ol>';

						$( '#backupcodes_codes' ).html( function(){ 
							$(this).html(lis);
							if ( confirm( '<?php echo esc_js( __( 'Do you want to download your Backup Codes as a txt file?', 'secupress' ) ); ?>' ) ) {
								let listeBackupCodes = document.getElementById('backupcodes_codes');
								let contenuTexte     = listeBackupCodes.innerText;
								let blob             = new Blob([contenuTexte], { type: 'text/plain' });
								let lien             = document.createElement('a');
								lien.href            = window.URL.createObjectURL(blob);
								lien.download        = 'backup_codes-<?php echo esc_js( sanitize_title( home_url() ) ); ?>.txt';
								lien.click();
							}
						} );
					}
				} );
			}
		} );

		// backupcodes_show_button
		$( '#backupcodes_show_button' ).on( "click", function( e ) {
			e.preventDefault();
			$( this ).hide();
			$( "#backupcodes_codes, #backupcodes_warning" ).show();
		} );

	} );
	</script>
	<?php } ?>
<?php
}

add_action( 'personal_options_update', 'secupress_otpauth_save_profile_settings');
/**
 * Save some meta data related to OTP
 *
 * @since 2.2.6
 * @author Julio Potier
 **/
function secupress_otpauth_save_profile_settings( $user_id ) {
    if ( ! current_user_can( 'edit_user', $user_id ) ) {
        return;
    }

	if ( isset( $_POST['double-auth_otp-mail_flag'] ) ) {
		secupress_otpauth_update_user_option( 'dont-otp-mail', ! isset( $_POST['double-auth_otp-mail'] ), $user_id );
	}

	if ( isset( $_POST['double-auth_otp-mobile_flag'] ) ) {
		secupress_otpauth_update_user_option( 'dont-otp-mobile', ! isset( $_POST['double-auth_otp-mobile'] ), $user_id );
	}

}

add_action( 'admin_post_secupress_otpauth_regen_secret', 'secupress_otpauth_regen_secret_ajax_post_cb' );
/**
 * Reset the OTP for the current_user
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_regen_secret_ajax_post_cb() {

	if ( ! isset( $_GET['_wpnonce'] ) ) {
		secupress_admin_die();
	}
	if ( ( isset( $_GET['user'] ) && ! wp_verify_nonce( $_GET['_wpnonce'], 'secupress_otpauth_regen_secret-' . (int) $_GET['user'] ) ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'secupress_otpauth_regen_secret' ) ) {
	}	

	$user_id = (int) $_GET['user'];

	$options = array( 'verified', 'skip', 'secret', 'timeout', 'lasttimeslot', 'lost', 'token', 'backupcodes', 'seed', 'qrcode-status' );
	array_map( function( $option ) use( $user_id ) {
		secupress_otpauth_delete_user_option( $option, $user_id );
	}, $options );

	wp_redirect( wp_get_referer() );
	die();
}

add_filter( 'manage_users_columns', 'secupress_otpauth_users_col' );
function secupress_otpauth_users_col( $columns ) {
	$columns['otp-auth'] = __( 'OTP Auth', 'secupress' );
	return $columns;
}

add_action( 'manage_users_custom_column', 'secupress_otpauth_users_col_content', 10, 3 );
function secupress_otpauth_users_col_content( $value, $column_name, $user_id ) {
	if ( 'otp-auth' !== $column_name ) {
		return $value;
	}
	$user = get_user_by( 'ID', $user_id );
	if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		return '——';
	}
	if ( secupress_otpauth_get_user_option( 'secret', $user_id ) &&
		 secupress_otpauth_get_user_option( 'verified', $user_id )
	) {
		return '<span class="dashicons dashicons-lock secupress-otp" title="' . __( 'OTP Auth Activated', 'secupress' ) . '"></span>';
	} else {
		return '<span class="dashicons dashicons-unlock secupress-otp" title="' . __( 'OTP Auth Not Activated', 'secupress' ) . '"></span>';
	}
}

add_action( 'wp_ajax_secupress_otpauth_renew_backup_codes',    'secupress_otpauth_renew_backup_codes_ajax_post_cb' );
add_action( 'admin_post_secupress_otpauth_renew_backup_codes', 'secupress_otpauth_renew_backup_codes_ajax_post_cb' );
/**
 * Set backup codes for a user
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_renew_backup_codes_ajax_post_cb( $uid = 0 ) {
	global $current_user;

	if ( $uid || isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'secupress_otpauth_renew_backup_codes' ) ) {
		$user_id = $uid ? $uid : $current_user->ID;
		$newkeys = secupress_generate_backupcodes();

		secupress_otpauth_update_user_option( 'backupcodes', $newkeys );

		if ( ! $uid ) {
			secupress_admin_send_response_or_redirect( $newkeys );
		}
	} else {
		secupress_admin_die();
	}
}

add_action( 'admin_notices', 'secupress_otpauth_warning_no_backup_codes' );
/**
 * This warnings are displayed when you ran out of auth backup codes
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_warning_no_backup_codes() {
	$codes = secupress_otpauth_get_user_option( 'backupcodes' );

	if ( is_array( $codes ) && count( array_filter( $codes ) ) === 1 ) {
		$message = sprintf( __( 'You are about to run out of backup codes! Only <strong>ONE</strong> remains Please <a href="%s">renew them</a>!', 'secupress' ), get_edit_profile_url( get_current_user_id() ) . '#otpauth_secret' );
		secupress_add_notice( $message, 'updated', 'backupcodes-1-' . date('mY') );
	} elseif ( is_array( $codes ) && ! count( array_filter( $codes ) ) ) {
		$message = sprintf( __( 'You have run out of backup codes! None are left. Please <a href="%s">renew them</a> now!', 'secupress' ), get_edit_profile_url( get_current_user_id() ) . '#otpauth_secret' );
		secupress_add_notice( $message, 'updated', 'backupcodes-0-' . date('mY') );
	}
}

add_action( 'admin_notices', 'secupress_otpauth_warning_not_set_yet' );
/**
 * This warnings are displayed when you did not yet generate a key
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_warning_not_set_yet() {
	global $current_user;
	if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $current_user ) ) {
		return;
	}
	if ( secupress_otpauth_get_user_option( 'verified' ) ) {
		return;
	}
	$message = sprintf( __( 'Your account requires a two-factor authentication using an OTP Authenticator App. Please <a href="%s">generate an application key</a>.', 'secupress' ), get_edit_profile_url( get_current_user_id() ) . '#otpauth_secret' );
	secupress_add_notice( $message, 'updated', 'need-otp-' . date('mY') );
}

add_action( 'admin_init', 'secupress_otpauth_redirect' );
/**
 * If your role need an OTP setup, displays it
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_redirect() {
	global $current_user, $pagenow;

	if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $current_user ) ) {
		return;
	}

	if ( 'admin-post.php' === $pagenow || 'admin-ajax.php' === $pagenow || ! is_user_logged_in() ||
		secupress_otpauth_get_user_option( 'verified' ) || (int) secupress_otpauth_get_user_option( 'skip' ) > time()
		) {
		return;
	}

	secupress_otpauth_delete_user_option( 'skip', $current_user->ID );

	$args              = [];
	$args['wpscripts'] = [ 'jquery' ];
	$args['head']      = '<style type="text/css">.hide-if-no-js{display:none} #confirmation_code{ text-align: center; font-size: xx-large; letter-spacing: 0.5em; font-family: Courier, Arial; } ol code{font-size: large}</style>' . "\n";
	$suffix            = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	$canskip           = true;
	$step              = ! isset( $_POST['step'] ) || ! is_int( absint( $_POST['step'] ) ) || absint( $_POST['step'] ) > 4 ? 0 : absint( $_POST['step'] );

	if ( ! $step ) {
		$submit_value = __( 'Continue &#8594;', 'secupress' );
	} else {
		$submit_value = __( 'Next &#8594;', 'secupress' );
	}

	ob_start();
	?>
	<form class="wrap" action="<?php echo secupress_get_current_url( 'raw' ); ?>" method="post">
		<h1><?php _e( 'Double Authentication Setting Required', 'secupress' ); ?></h1>
		<?php
		switch ( $step ) {
			// Step 1
			case 1:
				$otpauth_secret = secupress_generate_key( 16, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567' );
				secupress_otpauth_update_user_option( 'secret', $otpauth_secret, $current_user->ID );
				$args['wpscripts'] = [ 'jquery-core', 'secupress-jquery-qrcode', 'secupress-qrcode' ];
				$args['head']      = '<script type="text/javascript">jQuery(document).ready( function( $ ) { ' . "\n" .
					'var qrcode = "otpauth://totp/WordPress1:" + escape( "' . esc_js( $current_user->user_email ) . '" ) + "?secret=' . $otpauth_secret . '&issuer=' . esc_js( get_bloginfo( 'name' ) ) . '";' . "\n" .
					'$( ".hide-if-no-js" ).show();' . "\n" .
					'$( "#otpauth_qrcode" ).text( "" ).qrcode( { "text":qrcode } );' . "\n" .
					'$( "#otpauth_secret" ).text( "' . $otpauth_secret . '" ).css( "font-size", "2em" );' . "\n" .
					'$( ".hide-if-no-js" ).css( "display", "inline-block" );' .
					'} );</script>';
				?>
				<h2><?php printf( __( 'Step %d', 'secupress' ), $step ); ?></h2>
				<code id="otpauth_secret"><?php echo $otpauth_secret; ?></code>
				<div class="hide-if-js">
					<span id="otpauth_qrcode"></span>
					<br/>
					<span class="description">
						<p class="description hidden"><?php _e( 'You can <span class="hide-if-no-js">scan the QRCode or&nbsp;</span>use the provided key directly in your Mobile Application.', 'secupress' ); ?></p>
					</span>
				</div>
				<?php
			break;

			// Step 2 (backup codes)
			case 2:
				?>
				<h2><?php printf( __( 'Step %d', 'secupress' ), $step ); ?></h2>
				<?php

				_e( 'When your smartphone is unavailable or if you are unable to log in to your account using the OTP Authenticator, you will need to use one of these backup codes.', 'secupress' );

				secupress_otpauth_renew_backup_codes_ajax_post_cb( $current_user->ID );
				$otpauth_backupcodes = secupress_otpauth_get_user_option( 'backupcodes' );

				echo '<ol id="backupcodes_list">';
					foreach ( $otpauth_backupcodes as $bkcode ) {
						echo secupress_tag_me( secupress_code_me( $bkcode ), 'li' );
					}
				echo '</ol>';
				?>
				<button class="button button-secondary hide-if-no-js" type="button" onclick="secupress_download_backupcodes()"><?php _e( 'Download Backup Codes', 'secupress' ); ?></button>
				<script type="text/javascript">
					jQuery(document).ready( function( $ ) {
						$( ".hide-if-no-js" ).css( "display", "inline-block" );
					});
					function secupress_download_backupcodes() {
						let listeBackupCodes = document.getElementById('backupcodes_list');
						let contenuTexte     = listeBackupCodes.innerText;
						let blob             = new Blob([contenuTexte], { type: 'text/plain' });
						let lien             = document.createElement('a');
						lien.href            = window.URL.createObjectURL(blob);
						lien.download        = 'backup_codes-<?php echo esc_js( sanitize_title( home_url() ) ); ?>.txt';
						lien.click();
					}
				</script>
			    <br>
			    <?php
				_e( 'Keep them someplace accessible, like your wallet.<br>Each code can be used only once.', 'secupress' );
			break;

			// Step 3 (test & confirmation)
			case 3:
				$timeslot = false;
				if ( ! empty( $_POST['confirmation_code'] ) ) {
					$otpauth_secret = secupress_otpauth_get_user_option( 'secret' );
					$lasttimeslot      = secupress_otpauth_get_user_option( 'lasttimeslot' );
					$timeslot          = secupress_base32_verify( $otpauth_secret, $_POST['confirmation_code'], $lasttimeslot );
				}

				if ( $timeslot ) {
					secupress_otpauth_update_user_option( 'lasttimeslot', $timeslot, $current_user->ID );
					secupress_otpauth_update_user_option( 'verified', 1, $current_user->ID );

					$submit_value = __( 'Close', 'secupress' );
					$canskip      = false;

					?>
					<h2><?php _e( 'Done!', 'secupress' ); ?></h2>
					<img src="<?php echo plugins_url( '/inc/img/lock-done.png', __FILE__ ); ?>" alt="" role="presentation">
					<p><?php _e( 'You account is now using a Double Authentication, congratulations!', 'secupress' ); ?></p>
					<em><?php _e( 'If you need to set an application password, you can do it in your profile.', 'secupress' ); ?></em>
					<?php
				} else {
					$args['head'] .= '<script type="text/javascript">jQuery(document).ready( function( $ ) { var d = document.getElementById("confirmation_code"); d.focus(); d.select(); } );</script>' . "\n";

					?><h2><?php printf( __( 'Step %d', 'secupress' ), $step ); ?></h2>
					<p>
						<?php
						_e( 'Now you need to confirm your double authentication setting, please enter the code provided by your Mobile OTP Application', 'secupress' );
						?>
					</p>
					<p>
						<?php
						if ( isset( $_POST['confirmation_code'] ) ) {
							echo '<p class="error">';
							_e( 'Invalid Confirmation Code', 'secupress' );
							echo '</p>';
							$submit_value = _x( 'Retry &#8594;', 'verb', 'secupress' );
						}
						--$step;
						?>
						<input type="text" name="confirmation_code" id="confirmation_code" maxlength="6" placeholder="******" onkeypress="return event.charCode >= 48 && event.charCode <= 57 || event.charCode == 13" name="otp" id="otp" size="20" style="ime-mode: inactive;">
					</p>
					<?php
				}
			break;

			// Just redirect on the referer, it's ok now
			case 4:
				wp_safe_redirect( urldecode( wp_get_referer() ) );
				die();
			break;

			// No Step
			default:
				echo strip_tags( sprintf( __( 'Your account requires a double authentication using a Mobile Authenticator, you have to <a href="%s">generate an application key</a>.', 'secupress' ), get_edit_profile_url( get_current_user_id() ) . '#otpauth' ) );
				?>
				<p class="description"><?php
					printf(
						/* translators: 1 is a link to an Android app, 2 is a link to an iOS app */
						__( 'Get a Free <abbr title="One-Time Password">OTP</abbr> Mobile Application on %1$s and %2$s.', 'secupress' ),
						'<a href="' . __( 'https://play.google.com/store/apps/details?id=org.fedorahosted.freeotp', 'secupress' ) . '" target="_blank">Android</a>',
						'<a href="' . __( 'https://itunes.apple.com/us/app/freeotp-authenticator/id872559395?mt=8', 'secupress' ) . '" target="_blank">iOS</a>'
					);
				?></p>
				<?php
			break;
		}
		++$step;
		?>
		<input type="hidden" name="step" value="<?php echo $step; ?>">
		<input type="hidden" name="_wp_http_referer" value="<?php echo urlencode( esc_url_raw( secupress_get_current_url( 'raw' ) ) ); ?>">
		<?php
		submit_button( $submit_value );

		if ( $canskip ) {
			?>
			<p>
				<a href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=secupress_otpauth_skip' ), 'secupress_otpauth_skip' ) ); ?>">
					<small><?php _e( 'Skip this, i‘ll set it up later in my profile,<br/>please remind me in 7 days.', 'secupress' ); ?></small>
				</a>
			</p>
			<?php
		}
		?>
	</form>
	<?php
	$content = ob_get_contents();
	$title   = __( 'Double Authentication Account Settings', 'secupress' );
	ob_clean();

	// Display the correct informations
	secupress_action_page( $title, $content, $args );
}

add_action( 'admin_post_secupress_otpauth_skip', 'secupress_otpauth_skip_ajax_post_cb' );
/**
 * Let users skip the OTP setup :'(
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_skip_ajax_post_cb() {
	global $current_user;

	check_admin_referer( 'secupress_otpauth_skip' );

	array_map( 'secupress_otpauth_delete_user_option', array(
		'verified', 'skip', 'secret', 'timeout', 'lasttimeslot', 'lost', 'token', 'backupcodes', 'seed',
	) );

	secupress_otpauth_update_user_option( 'skip', time() + ( 7 * DAY_IN_SECONDS ) );

	wp_safe_redirect( wp_get_referer() );
	die();
}

/**************************/
/****** TOOLS *************/
/**************************/


/**
 * Creates backup codes
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @return (array)
 */
function secupress_generate_backupcodes() {
	/**
	 * Filter how many backup codes are generated, default 6, min 2, max 20.
	 * 
	 * @param (int) $n
	 */
	$n     = absint( apply_filters( 'secupress.plugins.otp-auth.backupcodes.n', 6 ) );
	$n     = secupress_minmax_range( $n, 2, 20 );
	$codes = [];
	while( count( $codes ) < $n ) {
		$codes[] = secupress_generate_key( 10, '1234567890ABCDEF' ); // Do not change the length 10
		$codes   = array_unique( $codes );
	}
	return $codes;
}

/**
 * Get a otpauth user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_otpauth_get_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$uid          = $uid ?? $current_user->ID;
	return get_user_option( 'secupress_otpauth_' . $option, $uid );
}

/**
 * Delete a otpauth user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_otpauth_delete_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$user_id      = $uid ? $uid : $current_user->ID;
	return delete_user_option( $user_id, 'secupress_otpauth_' . $option );
}

/**
 * Update a otpauth user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (string) $value
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_otpauth_update_user_option( $option, $value, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$user_id      = $uid ? $uid : $current_user->ID;
	return update_user_option( $user_id, 'secupress_otpauth_' . $option, $value );
}

/**
 * Get a otpauth option name
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * 
 * @return (string)
 **/
function secupress_otpauth_get_option_name( $option ) {
	global $wpdb;
	return sprintf( '%ssecupress_otpauth_%s', $wpdb->prefix, $option );
}

add_action( 'secupress.modules.deactivate_submodule_' . basename( __FILE__, '.php' ), 'secupress_otpauth_deactivation' );
add_action( 'secupress.plugins.deactivation', 'secupress_otpauth_deactivation' );
/**
 * On deactivation, delete anything related to OTP for any user
 *
 * @since 2.2.6
 * @author Julio Potier
 */
function secupress_otpauth_deactivation() {
	global $wpdb;

	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_lasttimeslot',  false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_backupcodes',   false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_verified',      false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_secret',        false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_timeout',       false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_token',         false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_seed',          false, true );
	delete_metadata( 'user', false, $wpdb->prefix . 'secupress_otpauth_qrcode-status', false, true );
}

add_filter( 'secupress.plugins.otp-auth.login.success', 'secupress_otpauth_delete_meta' );
add_filter( 'secupress.plugins.passwordless.autologin_success', 'secupress_otpauth_delete_meta' );
/**
 * Delete useless meta soon on login success
 *
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (WP_User) $user
 **/
function secupress_otpauth_delete_meta( $user ) {
	secupress_otpauth_delete_user_option( 'seed', $user->ID );
	secupress_otpauth_delete_user_option( 'qrcode-status', $user->ID );
	secupress_otpauth_delete_user_option( 'token', $user->ID );
	secupress_otpauth_delete_user_option( 'timeout', $user->ID );
}

add_filter( 'secupress.plugins.otp-auth.login.success', 'secupress_otpauth_send_alert_mail' );
add_filter( 'secupress.plugins.passwordless.autologin_success', 'secupress_otpauth_send_alert_mail' );
/**
 * Send an email when someones log ins an account
 *
 * @author Julio Potier
 * @since 2.2.6
 * @param (WP_User) $user
 **/
function secupress_otpauth_send_alert_mail( $user ) {
	if ( ! secupress_get_module_option( 'double-auth_otp-mail', 1, 'users-login' ) || secupress_otpauth_get_user_option( 'dont-otp-mail', $user->ID ) ) {
		return;
	}

	$subject = sprintf( __( '[%s] Logged in with OTP Authenticator', 'secupress' ), '###SITENAME###' );
	/**
	 * Filter the subject from the mail that gives the user a warning when OTP is used
	 * 
	 * @param (string) $subject
	 */
	$subject = apply_filters( 'secupress.plugins.otp-auth.warn_user.subject', $subject );

	$message = sprintf( __( 'Hello %1$s,<br>You or someone else has just logged into your account using the OTP method. If this was not you, we recommend that you <a href="###LOGINURL###">log into your account</a>, update your backup codes, and destroy all other active sessions. If you are unable to do so, please contact an administrator at ###ADMIN_EMAIL### for assistance.' ), $user->display_name );

	/**
	 * Filter the message from the mail that gives the user a warning when OTP is used
	 * 
	 * @param (string) $subject
	 */
	$message = apply_filters( 'secupress.plugins.otp-auth.warn_user.message', $message );

	add_filter(	'secupress.mail.headers', 'secupress_mail_html_headers' );
	return secupress_send_mail( $user->user_email, $subject, $message );
}
