<?php
/**
 * Module Name: PasswordLess Double Authentication
 * Description: When you try to log in, you'll receive an email containing a validation link, without clicking on it, you can't log in.
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.4.1
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

if ( ! secupress_passwordless_is_activated() ) {
	return;
}

// EMERGENCY BYPASS!
if ( defined( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) && constant( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) ) {
	return;
}

add_filter( 'authenticate', 'secupress_passwordless_login_authenticate', SECUPRESS_INT_MAX - 10, 3 );
/**
 * Send an email with the new unique login link.
 *
 * @since 2.2.1 If your role is affected by PasswordLess, force it.
 * @since 1.0
 * @author Julio Potier
 *
 * @param (object) $raw_user A WP_Error object if user is not correctly authenticated, a WP_User object if he is.
 * @param (string) $username A username or email address.
 * @param (string) $password User password.
 *
 * @return (object) A WP_Error or WP_User object.
 */
function secupress_passwordless_login_authenticate( 
	$raw_user, 
	$username, 
	#[\SensitiveParameter]
	$password
) {
	static $running = false;

	// Errors from other plugins.
	if ( is_wp_error( $raw_user ) ) {
		// Remove WP errors related to empty fields.
		unset( $raw_user->errors['empty_username'], $raw_user->errors['empty_password'] );

		if ( $raw_user->errors ) {
			// There are still errors, don't go further.
			$running = false;
			return $raw_user;
		}
	}

	if ( $running || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
		return $raw_user;
	}
	$running = true;

	// Make sure to process only credentials provided by the login form.
	$username = ! empty( $_POST['log'] ) && $username === $_POST['log'] ? $username : ''; // WPCS: CSRF ok.
	$password = ! empty( $_POST['pwd'] ) && $password === $_POST['pwd'] ? $password : ''; // WPCS: CSRF ok.

	if ( ! $username && ! $password || $username && $password ) {
		$running = false;
		if ( ! secupress_is_affected_role( 'users-login', 'double-auth', $raw_user ) ) {
			return $raw_user;
		} else {
			$error_msg = __( 'You cannot login using your password, please just fill your login or email address.', 'secupress' );
			$wp_error  = new WP_Error( 'authentication_failed', $error_msg );
			return $wp_error;
		}
	}
	if ( $username && isset( $_POST['passwordless_get_magik_link'] ) ) {
		if ( ! secupress_is_user( $raw_user ) ) {
			$by           = is_email( $username ) ? 'email' : 'login';
			$temp_user    = get_user_by( $by, $username );
			// if the login was a fake email like "foobar@example.com", try now with a login.
			if ( ! secupress_is_user( $temp_user ) && is_email( $username ) ) {
				$raw_user = get_user_by( 'login', $username );
			} else {
				$raw_user = $temp_user;
			}
		}
	}

	$result = secupress_passwordless_send_link( $raw_user, $_REQUEST );
	if ( $result['success'] ) {
		if ( $result['mailsent'] ) {
			// PasswordLess ok!
			wp_redirect( esc_url_raw( add_query_arg( 'action', 'passwordless_autologin', wp_login_url() ) ) );
			die();
		}
	} else {
		// id, username or email does not exist
		$errors = new WP_Error();
		// Display a vague error message.
		$errors->add( 'invalid_username', __( '<strong>Error</strong>: Invalid username or email.', 'secupress' ) );
		return $errors;
	}

	return $raw_user; // Should not happen
}

add_action( 'login_head', 'secupress_passwordless_buffer_start_login' );
/**
 * Start the buffer if we are on the login page.
 *
 * @since 1.0
 * @author Julio Potier
 */
function secupress_passwordless_buffer_start_login() {
	if ( ! isset( $_GET['action'] ) || 'login' === $_GET['action'] ) {
		ob_start( 'secupress_passwordless_hide_password_field_ob' );
	}
}

add_action( 'login_footer', 'secupress_passwordless_buffer_stop', 1000 );
/**
 * End the buffer if we are on the login page + not passwordless.
 *
 * @since 2.0 load gravatar if possible
 * @since 1.0
 * @author Julio Potier
 */
function secupress_passwordless_buffer_stop() {
	if ( isset( $_GET['action'] ) && 'login' !== $_GET['action'] && 'notpasswordless' !== $_GET['action'] ) {
		return;
	}
	?>
	<script>
	jQuery('#user_login').on('keyup',
		function() {
			var val = jQuery('#user_login').val();
			var md5email = jQuery('#md5email').length;
			var md5val   = md5email ? jQuery('#md5email').val() : '';
			if ( val.includes('@') ) { // Works only for email logins.
				var url  = jQuery('#loginform').val();
				if (url.indexOf('?') === -1) {
					url += '?';
				} else {
					url += '&';
				}
				url += 'action=md5&e=' + val;
				var ajax = jQuery.getJSON( url )
				.always( function(data) {
					jQuery('.login h1 a').css({ "background-image": 'url(https://0.gravatar.com/avatar/' + data.responseText + '?s:180)', "border-radius": '100%'});
				});
			} else if ( md5email && 32 === md5val.length ) {
				jQuery('.login h1 a').css({ "background-image": 'url(https://0.gravatar.com/avatar/' + md5val + '?s:180)', "border-radius": '100%'});
			} else {
				jQuery('.login h1 a').css({ "background-image": ''});
			}
		}
	).trigger('keyup');

	jQuery('.hide-if-js').hide();
	jQuery('#wp-submit').val(jQuery('#wp-submit').data('newvalue')).css('margin-bottom', '10px');
	jQuery('.forgetmenot').insertAfter('.submit');
	<?php 
	$sp_roles = secupress_get_module_option( 'double-auth_affected_role', [], 'users-login' );

	if ( count( $sp_roles ) ) { // Not everyone is affected, let the choice
	?>
	jQuery('#loginwithpassword').css('display', 'inline-block');
	jQuery('#loginwithpassword button:first').on('click', function(e){
		jQuery(this).parent().remove();
		jQuery('.forgetmenot').insertBefore('.submit');
		jQuery('.user-pass-wrap').show().find('input').prop('disabled', false).focus();
		jQuery('[name=passwordless_get_magik_link]').remove();
		jQuery('#wp-submit').val(jQuery('#wp-submit').data('origvalue')).css('margin-bottom', '0px');
	});
	<?php } ?>
	</script>
	<?php

	ob_end_flush();

	// Focus the password field.
	if ( ! isset( $_GET['action'] ) || 'notpasswordless' !== $_GET['action'] ) {
		return;
	}

	?>
<script type="text/javascript">
function secupress_attempt_focus() {
	setTimeout( function() {
		try {
			d = document.getElementById( 'user_pass' );
			d.focus();
			d.select();
		} catch( e ) {}
	}, 300 );
}

secupress_attempt_focus();

if ( typeof wpOnload === 'function' ) {
	wpOnload();
}
</script>
	<?php
}


/**
 * Alter the buffer content to hide the password label and field.
 *
 * @since 2.2.1 New UI/UX
 * @since 1.0
 * @author Julio Potier
 *
 * @param (string) $buffer Contains the login page between the action `login_head` and `login_footer`.
 *
 * @return (string)
 */
function secupress_passwordless_hide_password_field_ob( $buffer ) {
	$sp_roles = secupress_get_module_option( 'double-auth_affected_role', [], 'users-login' );

	$buffer   = str_replace( '<div class="user-pass-wrap">', '<div class="user-pass-wrap hide-if-js"><input type="hidden" name="passwordless_get_magik_link" value="1" />', $buffer );
	$buffer   = str_replace( 'value="' . esc_attr__( 'Log In' ) . '"', 'data-origvalue="' . esc_attr__( 'Log In' ) . '" data-newvalue="' . esc_attr__( 'Get my Magic Link', 'secupress' ) . '" value="' . esc_attr__( 'Log In' ) . '"', $buffer );
	if ( count( $sp_roles ) ) { // Not everyone is affected, let the choice
		$buffer   = str_replace( '</form>', sprintf( '<div id="loginwithpassword" style="display:none;line-height:2.35em">— %1$s <button type="button" class="button-secondary button-link" style="font-weight:600">%2$s</button></div>', __( 'or', 'secupress' ), __( 'Log In using a password', 'secupress' ) ) . '</form>', $buffer );
	}
	return $buffer;
}

add_filter( 'mepr-validate-login', 'secupress_passwordless_support_memberpress' );
/**
 * Support PasswordLess 2FA for MemberPress plugin
 *
 * @since 2.2 
 * @author Julio Potier 
 * @param (array) $errors
 * @return (array) $errors
 **/
function secupress_passwordless_support_memberpress( $errors ) {
	if ( ! isset( $_SERVER['REQUEST_METHOD'], $_POST['log'] ) || 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
		return $errors;
	}

	// Check for login by email address
	$by   = is_email( $_POST['log'] ) ? 'email' : 'login'; 
	$user = get_user_by( $by, $_POST['log'] );

	if ( ! is_wp_error( $user ) && secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		$message  = apply_filters( 'secupress.passwordless.memberpress_message', __( 'You cannot login using this page.', 'secupress' ) );
		$errors[] = $message;
	}

	return $errors;
}

add_filter( 'user_row_actions', 'secupress_passwordless_add_magiclink', 10, 2 );
/**
 * Add a "magic link" link for users
 *
 * @param (array) $actions
 * @param (WP_User) $user_object
 * @author Julio Potier
 * @since 2.2.1
 * @return (array) $actions
 **/
function secupress_passwordless_add_magiclink( $actions, $user_object, $force = false ) {
	if ( $force || ( get_current_user_id() !== $user_object->ID && current_user_can( 'edit_user', $user_object->ID ) ) ) {
		$actions['passwordless_magiclink'] = '<a class="resetpassword" href="' . wp_nonce_url( admin_url( 'admin-post.php?action=send_passwordless_magiclink&uid=' . $user_object->ID ), 'passwordless_magiclink_' . $user_object->ID ) . '">' . __( 'Send PasswordLess Magic Link', 'secupress' ) . '</a>';
	}
	return $actions;
}

add_action( 'admin_post_send_passwordless_magiclink', 'secupress_passwordless_send_magiclink' );
/**
 * Send the magic link even is the user is not affected by its role
 *
 * @since 2.2.1
 * @author Julio Potier
 **/
function secupress_passwordless_send_magiclink() {
	if ( ! isset( $_GET['_wpnonce'], $_GET['uid'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'passwordless_magiclink_' . $_GET['uid'] ) ) {
		wp_die( __( 'Magic Link not sent.', 'secupress' ) );
	}
	$result = secupress_passwordless_send_link( $_GET['uid'] );
	wp_redirect( admin_url( 'users.php?message=passwordless_' . [ 'ko', 'ok' ][ (int) $result['success'] ] ) );
	die();
}

add_action( 'admin_head-users.php', 'secupress_passwordless_messages' );
/**
 * Manage the feedback message from magic link admin area
 *
 * @since 2.2.1
 * @author Julio Potier
 **/
function secupress_passwordless_messages() {
	if ( ! isset( $_GET['message'] ) ) {
		return;
	}
	switch ( $_GET['message'] ) {
		case 'passwordless_ko':
			echo '<div class="error"><p>' . __( 'Magic Link not sent.', 'secupress' ) . '</p></div>';
		break;
		case 'passwordless_ok':
			echo '<div class="updated"><p>' . __( 'Magic Link has been sent.', 'secupress' ) . '</p></div>';
		break;
	}
}

add_filter( 'allow_password_reset', 'secupress_passwordless_disable_password_reset_for_some_users', 10, 2 );
/**
 * Prevent password reset for affected roles
 *
 * @since 2.4
 * @author Julio Potier
 * 
 * @param (bool) $allow
 * @param (int) $user_id
 * 
 * @return (bool)
 **/
function secupress_passwordless_disable_password_reset_for_some_users( $allow, $user_id ) {
	$user = secupress_get_user_by( $user_id );
	if ( secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		return false;
	}
	return $allow;
}
