<?php
/**
 * Module Name: Single Sign On (SSO)
 * Description: Users can be autoconnected on multiple MS sites
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.2.6
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

if ( ! is_multisite() ) {
	return;
}

add_action( 'wp_before_admin_bar_render', 'secupress_move_login_change_site_switcher_links' );
/**
 * Move Login SSO: Change the links in the admin menu bar
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 **/
function secupress_move_login_change_site_switcher_links() {
	global $wp_admin_bar;

	if ( ! is_multisite() ) {
		return;
	}

	$nodes           = $wp_admin_bar->get_nodes();
	$current_site_id = get_current_blog_id();
	$current_site    = get_site( $current_site_id );
	foreach ( $nodes as $id => $node ) {
		if ( empty( $node->href ) ) {
			continue;
		}
		$is_site_node          = ( 0 === stripos( $id, 'blog' ) );
		$is_network_admin_node = ( 0 === stripos( $id, 'network-admin' ) );
		if ( ! ( $is_site_node || $is_network_admin_node ) ) {
			continue;
		}
		if ( $current_site->domain === parse_url( $node->href, PHP_URL_HOST ) ) {
			continue;
		}
		$target_url_parts = wp_parse_url( $node->href );
		$current_user_id  = wp_get_current_user()->ID;
		$target_site      = get_site_by_path( $target_url_parts['host'], $target_url_parts['path'] );
		$nonce            = wp_create_nonce( 'secupress-sso-' . $current_site_id . '-' . $target_site->blog_id . '-' . $current_user_id );
		$node->href       = add_query_arg( [ 'secupress-sso' => $current_site_id, 'uid' => $current_user_id,'nonce' => $nonce ], $node->href );
		$wp_admin_bar->add_node( $node );
	}
}

add_action( 'secupress.plugins.loaded', 'secupress_move_login_receive_sso_request' );
/**
 * Move Login SSO: Initiate the workflow, on a target site that the user wants to log into.
 *
 * @since 2.2.6
 * @author Julio Potier
 *
*/
function secupress_move_login_receive_sso_request() {
	if ( ! is_multisite() || empty( $_GET['secupress-sso'] ) ) {
		return;
	}

	if ( is_user_logged_in() ) {
		wp_redirect( remove_query_arg( [ 'secupress-sso', 'uid', 'nonce' ] ) );
		die();
	}

	if ( empty( $_GET['nonce'] ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	if ( empty( $_GET['uid'] ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$coming_from = (int) $_GET['secupress-sso'];
	$sso_site = get_site( (int) $_GET['secupress-sso'] );
	if ( ! $sso_site ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$return_url = get_site_url() . remove_query_arg( [ 'secupress-sso', 'uid', 'nonce' ] );
	$next_url   = add_query_arg( [ 'secupress-sso-return-to' => $return_url, 'uid' => (int) $_GET['uid'], 'nonce' => $_GET['nonce'] ], get_site_url( $coming_from ) );
	
	wp_redirect( $next_url );
	die();
}

add_action( 'secupress.plugins.loaded', 'secupress_move_login_authorize_request' );
/**
 * Move Login SSO: Used on the authorizing site
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 */
function secupress_move_login_authorize_request() {
	if ( ! is_multisite() || empty( $_GET['secupress-sso-return-to'] ) ) {
		return;
	}
	if ( ! is_user_logged_in() ) {
		secupress_die( __( 'Single Sign On requires a logged-in user. Please log in first.', 'secupress' ) );
	}
	$current_user   = wp_get_current_user();
	$return_url     = esc_url_raw( $_GET['secupress-sso-return-to'] );

	if ( empty( $_GET['uid'] ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	if ( (int) $_GET['uid'] !== $current_user->ID ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$url_host           = parse_url( $_GET['secupress-sso-return-to'], PHP_URL_HOST );
	$requesting_site_id = get_blog_id_from_url( $url_host );
	if ( ! $requesting_site_id ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	if ( empty( $_GET['nonce'] ) || ! wp_verify_nonce( $_GET['nonce'], 'secupress-sso-' . get_current_blog_id() . '-' . $requesting_site_id . '-' . $_GET['uid'] ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$expiration_time = abs( max( 5, (int) apply_filters( 'secupress.plugins.move_login.expiration_time', 5 ) ) );
	$expires         = strtotime( "+$expiration_time minutes" );
	if ( empty( $current_user->user_pass ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$hash = implode( '||', [ (int) $current_user->ID, (int) $expires, $current_user->user_pass ] );
	$hash = hash_hmac( 'sha256', $hash, wp_salt() );
	if ( empty( $hash ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' ) . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$next_url = add_query_arg( [ 'secupress-sso-auth' => $hash, 'secupress-sso-user-id' => $current_user->ID, 'secupress-sso-expires' => $expires ], $return_url );
	wp_redirect( $next_url );
	exit();
}

add_action( 'secupress.plugins.loaded', 'secupress_move_login_receive_auth' );
/**
 * Move Login SSO: final step
 *
 * @since 2.2.6
 * @author Julio Potier
 *
*/
function secupress_move_login_receive_auth() {
	if ( ! is_multisite() ) {
		return;
	}

	$keys = [ 'secupress-sso-auth', 'secupress-sso-user-id', 'secupress-sso-expires' ];
	foreach ( $keys as $key ) {
		if ( empty( $_GET[ $key ] ) ) {
			return;
		}		
	}

	$final_destination = remove_query_arg( $keys );
	if ( is_user_logged_in() ) {
		wp_redirect( $final_destination );
		die();
	}

	$user_id       = (int) $_GET['secupress-sso-user-id'];
	$expires       = (int) $_GET['secupress-sso-expires'];
	$received_hash = $_GET['secupress-sso-auth'];
	if ( $expires < time() ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' )  . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$user_data      = get_userdata( $user_id );
	$expected_hash  = implode( '||', [ (int) $user_id, (int) $expires, $user_data->user_pass ] );
	$expected_hash  = hash_hmac( 'sha256', $expected_hash, wp_salt() );

	if ( empty( $expected_hash ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' )  . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}
	if ( ! hash_equals( $expected_hash, $received_hash ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' )  . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}
	if ( ! user_can( $user_id, 'read' ) ) {
		secupress_die( __( 'Single Sign On impossible right now, please retry.', 'secupress' )  . ' ERR#' . __LINE__, '', [ 'force_die' => true ] );
	}

	$user_data = get_userdata( $user_id );
	wp_set_current_user( $user_data->ID, $user_data->user_login );
	wp_set_auth_cookie( $user_data->ID );
	do_action( 'wp_login', $user_data->user_login, $user_data );

	wp_redirect( $final_destination );
	die();
}
