<?php
/**
 * Module Name: Strong Passwords
 * Description: Force the users to use a strong password.
 * Main Module: users_login
 * Author: SecuPress
 * Version: 2.3.19.1
 */

defined( 'SECUPRESS_VERSION' ) or die( 'Something went wrong.' );

// EMERGENCY BYPASS!
if ( defined( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) && constant( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) ) {
	return;
}

use ZxcvbnPhp\Zxcvbn;

/**
 * @since 2.3.19
 */
if ( secupress_is_pro() && secupress_get_module_option( 'password-policy_password_expiration', 0, 'users-login' ) ) {
	require_once( SECUPRESS_PRO_MODULES_PATH . 'users-login/plugins/password-expiration.php' );
}

add_action( 'show_user_profile',          'secupress_pro_strong_passwords_add_js' );
add_action( 'edit_user_profile',          'secupress_pro_strong_passwords_add_js' );
/**
 * Remove the checkbox "Confirm the usage of a weak password" on user edition page
 *
 * @since 2.3.19
 * @author Julio Potier
 * 
 * @param (WP_User) $user
 **/
function secupress_pro_strong_passwords_add_js( $user ) {
	wp_print_inline_script_tag( "jQuery('.pw-weak').remove();" );
	if ( ! secupress_passwordless_is_activated() || ! secupress_is_affected_role( 'users-login', 'password-policy', $user ) ) {
		return;
	}
	if ( ( ! defined( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) || ! constant( 'SECUPRESS_ALLOW_LOGIN_ACCESS' ) ) ) {
		$info = sprintf( '<em>' . __( 'This account is secured with %s. Password changes are not available.', 'secupress' ) . '</em>',
		secupress_get_capability() ? sprintf( '<a href="%s">%s</a>', secupress_admin_url( 'modules', 'users-login#row-double-auth_type' ), __( 'PasswordLess Double Authentication', 'secupress' ) ) : __( 'PasswordLess Double Authentication', 'secupress' ),
	);
		wp_print_inline_script_tag( 'jQuery(".wp-generate-pw").prop("disabled",true).after("<br><p>"+' . wp_json_encode( $info ) . '+"</p>");' );
	}
}

add_action( 'user_profile_update_errors', 'secupress_pro_strong_passwords_check_strength', 10, 3 );
/**
 * Check the new password strength.
 * If not strong, trigger an error.
 *
 * @since 1.0
 * @author Grégory Viguier
 *
 * @param (object) $errors WP_Error object, passed by reference.
 * @param (bool)   $update Whether this is a user update.
 * @param (object) $user   WP_User object, passed by reference.
 */
function secupress_pro_strong_passwords_check_strength( $errors, $update, $user ) {
	if ( empty( $_POST['pass1'] ) || empty( $user->user_pass ) || ! isset( $user->ID ) ) { // WPCS: CSRF ok.
		return;
	}
	// Skip strong password check if user is not affected by password policy
	if ( ! secupress_is_affected_role( 'users-login', 'password-policy', $user ) ) {
		return;
	}
	// Skip strong password check if user is using passwordless authentication
	if ( secupress_passwordless_is_activated() && secupress_is_affected_role( 'users-login', 'double-auth', $user ) ) {
		return;
	}

	$score = secupress_pro_strong_passwords_get_score( $user, $_POST['pass1'] );
	if ( 4 !== $score ) {
		$error = '';
		$tips  = secupress_pro_strong_passwords_get_tips( $user->user_pass );
		if ( ! empty( $tips ) ) {
			$error .= __( 'Please choose a stronger password.', 'secupress' );
			$error .= '<br>' . sprintf( _n( 'Improve it with this tip: %s.', 'Improve it with these tips: %s.', count( $tips ), 'secupress' ), wp_sprintf_l( '%l', $tips ) );
		}
		$errors->add( 'weak_password', $error, array( 'form-field' => 'pass1' ) );
	}
}

add_action( 'authenticate', 'secupress_pro_strong_passwords_on_login', SECUPRESS_INT_MAX - 5, 3 );
/**
 * Check the password force on login
 *
 * @since 2.3.19 Revamp with secupress_login_page
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (mixed) $raw_user
 * @param (string) $username
 * @param (string) $password The users's password in clear
 * 
 * @return (mixed) $raw_user Can also display a form to ask a new strong password
 **/
function secupress_pro_strong_passwords_on_login( $raw_user, $username, $password ) {
	global $sp_action;

	if ( 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
		return $raw_user;
	}

	$tmp_user   = secupress_get_user_by( $username );
	if ( ! secupress_is_user( $tmp_user ) ) {
		return $raw_user; // Wrong password return is here
	}

	// Skip if user is not affected by password policy
	if ( ! secupress_is_affected_role( 'users-login', 'password-policy', $tmp_user ) ) {
		return $raw_user;
	}

	// Skip if user is using passwordless authentication
	if ( secupress_passwordless_is_activated() && secupress_is_affected_role( 'users-login', 'double-auth', $tmp_user ) ) {
		return $raw_user;
	}

	$update_user_pass = false;
	$user_id          = $tmp_user->ID;
	// We may need to switch to user locale now
	if ( $user_id ) {
		switch_to_locale( get_user_locale( $user_id ) );
	}
	
	if ( ! is_a( $raw_user, 'WP_User' ) ) {
		$tmp_user   = secupress_get_user_by( $username );
		if ( ! secupress_is_user( $tmp_user ) ) {
			return $raw_user; // Wrong password return is here
		}
		$user_token = secupress_pro_strong_passwords_get_user_option( 'token', $tmp_user->ID );
		$user_token = ! $user_token ? '' : $user_token;
		$user_db    = secupress_pro_strong_passwords_get_user_from_meta_value( 'token', $password );
		$time       = secupress_pro_strong_passwords_get_user_option( 'timeout', $tmp_user->ID );
		
		if ( ! secupress_is_user( $tmp_user ) || ! $user_db || $user_db !== $tmp_user->ID || ! hash_equals( $user_token, $password ) ) {
			return $raw_user; // Wrong password return is here
		}

		if ( $time < time() ) {
			$error = __( 'Too much time has elapsed between the first step and now.', 'secupress' );
			return new WP_Error( 'expired_time', $error );
		}

		$update_user_pass = true;
		$raw_user         = $tmp_user;
		$user_id          = $raw_user->ID;
		// We may need to do it here now.
		switch_to_locale( get_user_locale( $user_id ) );
		unset( $tmp_user );
	}

	global $sp_action;
	$sp_action        = 'new-password';
	$post_param       = "secupress-strong-password-{$sp_action}";
	$nonce_action     = "{$post_param}-{$raw_user->ID}";
	$error            = '';

	$stop = 0;
	// Password expiration
	if ( ! $update_user_pass && secupress_get_module_option( 'password-policy_password_expiration', 0, 'users-login' ) > 0 ) {
		if ( secupress_pro_password_expiration_is_user_password_expired( $raw_user ) ) {
			$message = __( 'Your current password has expired.', 'secupress' );
			$context = 'password-expiration';
			++$stop;
		}
	}

	// During the password update, overwrite the $password
	if ( $update_user_pass && isset( $_POST['pass1'], $_POST['sp_action'] ) && $sp_action === $_POST['sp_action'] ) {
		$password         = $_POST['pass1'];
		$update_user_pass = true;
	}
	$score = secupress_pro_strong_passwords_get_score( $raw_user, $password );
	// Password is weak
	if ( 4 !== $score ) {
		$message = __( 'Your current password strength is too weak.', 'secupress' );
		$context = 'strong-passwords';
		++$stop;
	}

	// Everything is ok, let it go.
	if ( ! $update_user_pass && ! $stop ) {
		return $raw_user;
	}

	$new_token  = wp_hash( wp_generate_password( 32, false ) );
	secupress_pro_strong_passwords_update_user_option( 'token', $new_token, $user_id );
	secupress_pro_strong_passwords_update_user_option( 'timeout', time() + ( 10 * MINUTE_IN_SECONDS ), $user_id );
	
	// A new password is submitted.
	if ( isset( $_POST['pass1'], $_POST['sp_action'] ) && $sp_action === $_POST['sp_action'] ) {

		if ( empty( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], $nonce_action ) ) {
			secupress_die( __( 'Something went wrong.', 'secupress' ), '', [ 'force_die' => true, 'context' => $context, 'attack_type' => 'login' ] );
		}

		$new_pass = $_POST['pass1'];
		if ( empty( $new_pass ) ) {
			$error = __( 'Password required', 'secupress' );
		} elseif ( wp_check_password( $new_pass, $raw_user->user_pass, $user_id ) ) {
			$error = __( 'Please enter a new password.', 'secupress' );
		} else {
			$score = secupress_pro_strong_passwords_get_score( $raw_user, $new_pass );
			if ( 4 !== $score ) {
				$error = __( 'Please choose a stronger password.', 'secupress' );
				$tips  = secupress_pro_strong_passwords_get_tips( $new_pass );
				if ( ! empty( $tips ) ) {
					$error .= '<br>' . sprintf( _n( 'Improve it with this tip: %s.', 'Improve it with these tips: %s.', count( $tips ), 'secupress' ), wp_sprintf_l( '%l', $tips ) );
				}
			}
			wp_set_password( $new_pass, $user_id );
			// WP Behavior
			update_user_meta( $user_id, 'default_password_nag', false );
			do_action( 'after_password_reset', $raw_user, $new_pass );
			
			// Send the new login by email.
			secupress_pro_strong_passwords_new_pass_notification( $raw_user );
			secupress_pro_strong_passwords_delete_user_option( 'token' );
			secupress_pro_strong_passwords_delete_user_option( 'timeout' );

			// Kill session.
			wp_clear_auth_cookie();
			wp_destroy_current_session();

			// Redirect the user to the login page.
			$redirect  = strpos( secupress_get_current_url( 'raw' ), wp_login_url() ) === false ? secupress_get_current_url( 'raw' ) : admin_url();
			$login_url = wp_login_url( $redirect, true );
			$login_url = add_query_arg( 'secupress-relog', $sp_action, $login_url );
			$login_url = add_query_arg( 'wp_lang', get_user_locale(), $login_url );
			wp_redirect( esc_url_raw( $login_url ) );
			die();
		}
	}

	ob_start();
	$css_rules = "\n" .
'<style type="text/css">
#pass1-text{display:none;}
.pw-weak{display:none!important;}
#secupress-pass1{display:inline-block;font-size:1.5em;}
#pass1-text{padding-right:2.5rem;}
#secupress-pass-strength-result{position:relative;top:-11px;width:100%;padding:0px;font-family:monospace;height:1.5em;font-weight:600}
.wp-generate-pw{width:100%;}
</style>' . "\n";
	
	$message .= '<br>' . _n( 'You must update it to successfully log in.', 'You must update them to successfully log in.', 1, 'secupress' ); // 1 : one password

	$wp_errors = new WP_Error();
	$wp_errors->add( "{$sp_action}-message", $message, 'message' );
	if ( $error ) {
		$wp_errors->add( "{$sp_action}-error", $error, 'error' );
		add_action( 'secupress_login_page.shake_js', '__return_true' );
	}
	// The form.
	ob_start();
	?>
	<form name="loginform" id="loginform" action="" method="post">
		<table class="form-table login-page <?php echo is_rtl() ? 'rtl' : 'ltr'; ?>" style="width: 100%">
			<tr id="password" class="secupress-user-pass1-wrap"><td>
			<p>
				<button type="button" class="button button-primary wp-generate-pw hide-if-no-js"><?php _e( 'Set New Password', 'secupress' ); ?></button>
				<div class="wp-pwd">
					<span class="password-input-wrapper">
						<input type="password" name="pass1" id="secupress-pass1" class="regular-text" value="" autocomplete="off" data-pw="<?php echo esc_attr( wp_generate_password( 12 ) ); ?>" aria-describedby="secupress-pass-strength-result" />
					</span>
					<button type="button" class="button button-secondary wp-hide-pw hide-if-no-js" data-toggle="0" aria-label="<?php esc_attr_e( 'Hide password', 'secupress' ); ?>">
						<span class="dashicons dashicons-visibility" aria-hidden="true"></span>
					</button>
					<div style="display:none" id="secupress-pass-strength-result" aria-live="polite"></div>
				</div>
				<p class="pw-weak hide-if-js" tabindex="0"><?php _e( 'You must choose a strong password.', 'secupress' ); ?></p>
				<?php
				if ( ! function_exists( 'submit_button' ) ) {
					require_once ABSPATH . 'wp-admin/includes/template.php';
				}
				submit_button( _x( 'Renew', 'verb', 'secupress' ), 'primary', 'thesubmit' );
				wp_nonce_field( $nonce_action );
				?>
			</p>
			</td></tr>
			<tr class="secupress-user-pass2-wrap hidden"><td>
				<input name="pass2" type="password" id="secupress-pass2" class="regular-text" value="" autocomplete="off" />
				<input type="hidden" name="log" value="<?php echo esc_attr( $username ); ?>" />
				<input type="hidden" name="pwd" value="<?php echo esc_attr( $new_token ); ?>" />
				<input type="hidden" name="sp_action" value="<?php echo esc_attr( $sp_action ); ?>">
			</td></tr>
		</table>
	</form>
	<?php

	echo $css_rules;
	$content = ob_get_contents();
	ob_end_clean();
	unset( $raw_user );
	if ( ! secupress_is_soft_request() ) {
		wp_send_json_error( [ 'error' => $error, 'message' => $message ] );
	} else {
		secupress_pro_strong_passwords_scripts();
		secupress_pro_strong_passwords_styles();

		secupress_login_page( __( 'Please renew your password', 'secupress' ), $content, $wp_errors, $user_id );
	}
}

/** --------------------------------------------------------------------------------------------- */
/** TOOLS ======================================================================================= */
/** ----------------------------------------------------------------------------------------------*/

/**
 * Returns the zxcvbn score of a password in PHP
 *
 * @since 2.3.14 SensitiveParameter
 * @author Julio Potier
 * @since 1.0
 * @author Gregory Viguier
 * 
 * @param (WP_User) $user
 * @param (string) $password
 * 
 * @return 
 **/
function secupress_pro_strong_passwords_get_score( 
	$user, 
	#[\SensitiveParameter] 
	$password
) {
	// Collect all the strings we want to avoid
	$disallowed  = [];
	$user_data   = [ 'user_login' => 1, 'first_name' => 1, 'last_name' => 1, 'nickname' => 1, 'display_name' => 1, 'user_email' => 1, 'user_url' => 1, 'description' => 1 ];
	$user_data   = array_intersect_key( (array) $user, $user_data );
	$user_data[] = get_bloginfo( 'name' );
	$user_data[] = get_bloginfo( 'admin_email' );

	// Strip out non-alphanumeric characters and convert each word to an individual entry.
	foreach ( $user_data as $i => $data ) {
		if ( $data ) {
			$data       = preg_replace( '@\W@', ' ', $data );
			$data       = explode( ' ', $data );
			$disallowed = array_merge( $disallowed, $data );
		}
	}

	if ( $disallowed ) {
		$disallowed = array_filter( $disallowed, function( $data ) {
			return $data !== '' && mb_strlen( $data ) >= 4;
		});
	}

	if ( $disallowed ) {
		$disallowed = array_flip( array_flip( $disallowed ) );
	}

	// Test the password.
	spl_autoload_register( 'secupress_pro_strong_passwords_autoload' );

	$zxcvbn   = new Zxcvbn();
	$strength =  $zxcvbn->passwordStrength( $password, $disallowed ); // WPCS: CSRF ok.

	return $strength['score'];
}

/**
 * Get translated tips to help users to improve their password strength
 *
 * @since 2.3.19
 * @author Julio Potier
 * 
 * @param (string) $password
 * 
 * @return (array) $tips
 **/
function secupress_pro_strong_passwords_get_tips( $password ) {
	$tips = [];
	if ( 0 === preg_match( '/\p{Lu}/u', $password ) ) {
		$tips['upper']  = __( 'uppercase letters', 'secupress' );
	}
	if ( 0 === preg_match( '/\p{Ll}/u', $password ) ) {
		$tips['lower']  = __( 'lowercase letters', 'secupress' );
	}
	if ( 0 === preg_match( '/\d/', $password ) ) {
		$tips['number'] = __( 'numbers', 'secupress' );
	}
	if ( 0 === preg_match( '/[^\\p{L}\\p{N} ]/u', $password ) ) {
		$tips['symbol'] = __( 'symbols', 'secupress' );
	}
	if ( $password !== preg_replace('/(.)\1{2,}/', '$1$1', $password ) ) {
		$tips['3dup']   = __( 'no consecutive duplicate letters', 'secupress' );
	}
	if ( strtolower( $password ) !== strtolower( implode( '', array_unique( str_split( $password ) ) ) ) ) {
		$tips['more']   = __( 'more unique letters', 'secupress' );
	}
	if ( mb_strlen( $password, 'UTF-8' ) < 12 ) {
		$tips['length'] = __( 'at least 12 characters', 'secupress' );
	}
	return $tips;
}

/**
 * Send an email notification to a user with their new password
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (int|object) $user A user ID or a user object.
 */
function secupress_pro_strong_passwords_new_pass_notification( $user ) {
	$user            = secupress_is_user( $user ) ? $user : get_userdata( $user );
	$switched_locale = switch_to_user_locale( $user->ID );
	/* translators: Do not translate USERNAME, ADMIN_EMAIL, EMAIL, SITENAME, SITEURL: those are placeholders. */
	$pass_change_text = __(
		'Hi ###USERNAME###,

This notice confirms that your password was changed on ###SITENAME###.

If you did not change your password, please contact the Site Administrator at
###ADMIN_EMAIL###

This email has been sent to ###EMAIL###

Regards,
All at ###SITENAME###
###SITEURL###' );

	$pass_change_email = [
		'to'      => $user->user_email,
		'subject' => sprintf( __( '[%s] Password Changed', 'secupress' ), '###SITENAME###' ),
		'message' => $pass_change_text,
		'headers' => [],
	];
	$pass_change_email = apply_filters( 'secupress.plugins.strongpassword.change_email', $pass_change_email, $user, $user );
	$pass_change_email['message'] = str_replace( '###USERNAME###', $user->user_login, $pass_change_email['message'] );
	$pass_change_email['message'] = str_replace( '###EMAIL###', $user->user_email, $pass_change_email['message'] );

	secupress_send_mail( $pass_change_email['to'], $pass_change_email['subject'], $pass_change_email['message'], $pass_change_email['headers'] );

	restore_previous_locale();
}

/**
 * Autoloader for `Zxcvbn`.
 *
 * @since 1.0
 * @author Grégory Viguier
 *
 * @param (string) $class_id The class identifier, like `ZxcvbnPhp\Zxcvbn`.
 */
function secupress_pro_strong_passwords_autoload( $class_id ) {
	$class_id  = str_replace( '\\', '/', $class_id );
	$class_id  = trim( $class_id, '/' );
	$file_path = __DIR__ . '/inc/php/' . $class_id . '.php';

	if ( file_exists( $file_path ) ) {
		require_once( $file_path );
	}
}

/**
 * Get a strong passwords option name
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * 
 * @return (string)
 **/
function secupress_pro_strong_passwords_get_option_name( $option ) {
	global $wpdb;
	return sprintf( '%ssecupress_strong_passwords_%s', $wpdb->prefix, $option );
}

/**
 * Get a user_id from a meta value, only for otp auth
 *
 * @since 2.2.6
 * @author Julio Potier
 *
 * @param (string) $key
 * @param (string) $value
 *
 * @return (array)
 */
function secupress_pro_strong_passwords_get_user_from_meta_value( $key, $value ) {
	global $wpdb;
	$res = secupress_cache_data( "$key|$value" );
	if ( $res ) {
		return $res;
	}
	$res = (int) $wpdb->get_var( $wpdb->prepare( "SELECT user_id FROM $wpdb->usermeta WHERE meta_key = %s AND meta_value = %s", secupress_pro_strong_passwords_get_option_name( $key ), $value ) );
	secupress_cache_data( "$key|$value", $res );
	return $res;
}

/**
 * Get a strong password user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_strong_passwords_get_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$uid          = $uid ?? $current_user->ID;
	return get_user_option( 'secupress_strong_passwords_' . $option, $uid );
}

/**
 * Delete a strong password user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_strong_passwords_delete_user_option( $option, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$user_id      = $uid ? $uid : $current_user->ID;
	return delete_user_option( $user_id, 'secupress_strong_passwords_' . $option );
}

/**
 * Update a strong password user option easily
 * 
 * @since 2.2.6
 * @author Julio Potier
 * 
 * @param (string) $option
 * @param (string) $value
 * @param (int) $uid 0 = current_user
 * 
 * @return (string)
 **/
function secupress_pro_strong_passwords_update_user_option( $option, $value, $uid = 0 ) {
	$current_user = wp_get_current_user();
	$user_id      = $uid ? $uid : $current_user->ID;
	return update_user_option( $user_id, 'secupress_strong_passwords_' . $option, $value );
}

/** --------------------------------------------------------------------------------------------- */
/** JS / CSS ==================================================================================== */
/** ----------------------------------------------------------------------------------------------*/

/**
 * Enqueue plugin JS scripts.
 *
 * @since 1.0
 * @author Grégory Viguier
 */
function secupress_pro_strong_passwords_scripts() {
	global $wp_scripts;

	// Remove WP "zxcvbn" and "password-strength-meter".
	wp_deregister_script( 'zxcvbn-async' );
	wp_deregister_script( 'password-strength-meter' );

	// Keep the original "user-profile" because it does other things than dealing with the passwords. We'll change its dependencies.
	if ( ! empty( $wp_scripts->registered['user-profile'] ) ) {
		$deps   = array_flip( $wp_scripts->registered['user-profile']->deps );
		unset( $deps['password-strength-meter'] );
		$deps   = array_flip( $deps );
		$deps[] = 'secupress-password-strength-meter';
		$wp_scripts->registered['user-profile']->deps = $deps;
	}

	// Our scripts.
	$suffix   = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	$version  = $suffix ? SECUPRESS_PRO_VERSION : time();
	$base_url = SECUPRESS_PRO_URL . 'modules/users-login/plugins/inc/';

	// "zxcvbn".
	wp_enqueue_script( 'secupress-zxcvbn-async', "{$base_url}js/zxcvbn-async{$suffix}.js", array(), '4.3.0' );

	wp_localize_script( 'secupress-zxcvbn-async', '_zxcvbnSettings', array(
		'src' => "{$base_url}js/zxcvbn.min.js?ver=4.4.2&nocache=1",
	) );

	// "password-strength-meter".
	wp_enqueue_script( 'secupress-password-strength-meter', "{$base_url}js/password-strength-meter{$suffix}.js", array( 'jquery', 'secupress-zxcvbn-async' ), $version, true );

	wp_localize_script( 'secupress-password-strength-meter', 'pwsL10n', array(
		'short'    => _x( 'Very weak', 'password strength', 'secupress' ),
		'bad'      => _x( 'Weak', 'password strength', 'secupress' ),
		'good'     => _x( 'Medium', 'password strength', 'secupress' ),
		'strong'   => _x( 'Strong', 'password strength', 'secupress' ),
		'mismatch' => _x( 'Mismatch', 'password mismatch', 'secupress' ),
	) );

	// "user-profile".
	wp_enqueue_script( 'secupress-user-profile', "{$base_url}js/user-profile{$suffix}.js", array( 'jquery', 'secupress-password-strength-meter', 'wp-util' ), $version, true );

	wp_localize_script( 'secupress-user-profile', 'userProfileL10n', array(
		'warn'     => __( 'Your new password has not been saved.', 'secupress' ),
		'show'     => _x( 'Show', 'verb', 'secupress' ),
		'hide'     => _x( 'Hide', 'verb', 'secupress' ),
		'cancel'   => _x( 'Cancel', 'verb', 'secupress' ),
		'ariaShow' => __( 'Show password', 'secupress' ),
		'ariaHide' => __( 'Hide password', 'secupress' ),
	) );
}


/**
 * Enqueue plugin styles.
 *
 * @since 1.0
 * @author Grégory Viguier
 */
function secupress_pro_strong_passwords_styles() {
	$suffix   = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
	$version  = $suffix ? SECUPRESS_PRO_VERSION : time();
	$base_url = SECUPRESS_PRO_URL . 'modules/users-login/plugins/inc/';

	wp_enqueue_style(  'secupress-strong-passwords', "{$base_url}css/strong-passwords{$suffix}.css", [ 'forms' ], $version );
	wp_enqueue_script( 'secupress-strong-passwords', "{$base_url}js/strong-passwords{$suffix}.js", [ 'jquery' ], $version );
	// Needed for the action page on login
	wp_register_style( 'secupress-wordpress-css', SECUPRESS_ADMIN_CSS_URL . 'secupress-wordpress' . $suffix . '.css', [], $version );
}
