<?php
/**
 * Module Name: User Creation Protection
 * Description: Adds metadata to user creation and prevents login without metadata.
 * Version: 2.2.6
 * Author: Julio Potier
 * Author URI: https://secupress.me
 */

define( 'SECUPRESS_ADMIN_IDS',    'secupress_admin_ids' );
define( 'SECUPRESS_SUPER_ADMINS', 'secupress_super_admins' );

class SecuPress_User_Protection {

	/**
	 * Initialize the module
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function __construct() {
		add_action( 'secupress.pro.plugins.activation',                                       [ $this, 'activate_user_protection' ] );
		add_action( 'secupress.modules.activate_submodule_' . basename( __FILE__, '.php' ),   [ $this, 'activate_user_protection' ] );
		add_action( 'secupress.pro.plugins.deactivation',                                     [ $this, 'deactivate_user_protection' ] );
		add_action( 'secupress.modules.deactivate_submodule_' . basename( __FILE__, '.php' ), [ $this, 'deactivate_user_protection' ] );
		

		remove_all_filters( 'views_users' );
		remove_all_filters( 'pre_user_query' );

		add_filter( 'map_meta_cap',                    [ $this, 'remove_change_role_select' ], 10, 2 );
		add_action( 'auth_cookie_valid',               [ $this, 'force_disconnect_fake_pending_user' ], 10, 2 );
		add_filter( 'bulk_actions-users',              [ $this, 'remove_bulk_select' ] );
		add_action( 'user_register',                   [ $this, 'update_user_and_send_validation_email' ] );
		add_action( 'register_new_user',               [ $this, 'update_user_and_send_validation_email' ] );
		add_action( 'set_user_role',                   [ $this, 'update_user_and_send_validation_email' ], 10, 2 );
		add_action( 'remove_user_role',                [ $this, 'update_admin_ids_on_remove_user_role' ], 10, 2 );
		add_filter( 'authenticate',                    [ $this, 'prevent_login_without_metadata' ], 22 );
		add_filter( 'authenticate',                    [ $this, 'prevent_login_for_pending_users' ], 99 );
		add_filter( 'lostpassword_user_data',          [ $this, 'prevent_fake_users_password_reset' ], 1 );
		add_action( 'pre_get_users',                   [ $this, 'filter_fake_users' ] );
		add_filter( 'pre_count_users',                 [ $this, 'count_users' ], 10, 3 );
		add_action( 'pre_get_posts',                   [ $this, 'exclude_posts_by_fake_users' ] );
		add_action( 'pre_get_comments',                [ $this, 'exclude_comments_by_fake_users' ] );
		add_filter( 'getarchives_where',               [ $this, 'exclude_archives_by_fake_users' ] );
		add_filter( 'get_terms_args',                  [ $this, 'exclude_terms_by_fake_users' ], 10, 2 );
		add_filter( 'get_terms_args',                  [ $this, 'exclude_terms_by_fake_users' ], 10, 2 );
		add_filter( 'user_row_actions',                [ $this, 'add_user_status_action_link' ], 10, 2 );
		add_action( 'admin_head-users.php',            [ $this, 'highlight_users_without_metadata' ] );
		add_filter( 'views_users',                     [ $this, 'add_fake_and_pending_users_view' ] );
		add_action( 'admin_menu',                      [ $this, 'add_fake_users_counter' ] );
		add_filter( 'manage_users_columns',            [ $this, 'add_status_column' ] );
		add_filter( 'manage_users_custom_column',      [ $this, 'status_column_content' ], 10, 3 );
		add_filter( 'admin_post_secupress_activate',   [ $this, 'activate_pending_user' ] );
		add_filter( 'admin_post_secupress_downgrade',  [ $this, 'downgrade_pending_user' ] );
		// Multisite
		if ( is_multisite() ) {
			add_filter( 'pre_site_option_site_admins',     [ $this, 'pre_site_option_site_admins' ] );
			add_filter( 'default_site_option_site_admins', [ $this, 'pre_site_option_site_admins' ] );
			add_action( 'grant_super_admin',               [ $this, 'do_not_grant_or_revoke_super_admin' ] );
			add_action( 'revoke_super_admin',              [ $this, 'do_not_grant_or_revoke_super_admin' ] );
			add_action( 'load-user-edit.php',              [ $this, 'remove_grant_checkbox' ] );
			add_filter( 'wpmu_users_columns',              [ $this, 'role_network_users_columns' ] );
		}
		add_action( 'manage_users_custom_column',      [ $this, 'role_network_users_custom_column' ], 10, 3);
	}

	/**
	 * Add the "Role" columns
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (array) $columns
	 * 
	 * @return (array) $columns
	 */
	public function role_network_users_columns( $columns ) {
		$columns['role'] = __( 'Role', 'secupress' );
		return $columns;
	}

	/**
	 * Fill the Role column
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (string) $value
	 * @param (string) $column_name
	 * @param (int) $user_id
	 */
	public function role_network_users_custom_column( $value, $column_name, $user_id ) {
		if ( 'role' !== $column_name ) {
			return $value;
		}
		$role_names = [];
		$user_data  = get_userdata( $user_id );
		foreach( $user_data->roles as $user_site_role ) {
			$role_names[] = secupress_translate_user_role( wp_roles()->roles[ $user_site_role ]['name'], 'secupress' );
		}
		echo wp_sprintf_l( '%l', array_unique( $role_names ) );
	}

	/**
	 * Remove the grant super admin checkbox in users.php
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 */
	public function remove_grant_checkbox(){
		global $super_admins; 
		// Yep, just set the value, it's done.
		$super_admins = get_site_option( SECUPRESS_SUPER_ADMINS );
	}

	/**
	 * Update the user_status column for a user
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (int) $user_id
	 * @param (int) $user_status
	 * @param (int) $user_activation_key
	 */
	public function update_pending_status( $user_id, $user_status, $user_activation_key = false ){
		global $wpdb;

		$what = [ 'user_status' => (int) $user_status ];
		if ( false !== $user_activation_key ) {
			$what = [ 'user_activation_key' => $user_activation_key ];
		}
		$wpdb->update( $wpdb->users, $what, [ 'ID' => (int) $user_id ] ); // do not use wp_update_user() here, loop.
		/**
		 * Run an action when a user_status has been updated
		 *
		 * @since 2.2.6
		 * @author Julio Potier
		 * 
		 * @param (int) $user_id
		 * @param (int) $user_status
		 * @param (string) $user_activation_key
		 */
		do_action( 'secupress.plugins.user_creation_protection.status_updated', $user_id, $user_status, $user_activation_key );
	}

	public function update_admin_ids_on_remove_user_role( $user_id, $old_role ) {
		global $wpdb;

		if ( 'administrator' === $old_role ) {
			$user = secupress_get_user_by( $user_id );
			if ( $this->is_pending_user( $user ) ) {
				$this->update_pending_status( $user_id, 0 );
			}
			update_option( SECUPRESS_ADMIN_IDS, secupress_get_admin_ids_by_capa(), 'no' );
		}
	}

	/**
	 * Update the user status, send mail if ADMIN is the desired role + SP key when creating user
	 *
	 * @since 2.2.6
	 * @author Julio potier
	 * 
	 * @param (int)    $user_id
	 * @param (string) $role
	 */
	public function update_user_and_send_validation_email( $user_id, $role = '' ) {
		global $wpdb;
		static $done;

		if ( $user_id === $done ) {
			return;
		}
		
		$done   = $user_id;
		$user   = secupress_get_user_by( $user_id );
		$modulo = secupress_get_option( 'secupress_user_protection_modulo' );
		$seed   = secupress_get_option( 'secupress_user_protection_seed' );
		$role   = $role ?? reset( $user->roles );
		update_user_meta( $user_id, SECUPRESS_USER_PROTECTION, ( $user_id * $seed ) % $modulo );

		if ( 'administrator' === $role ) {

			// Generate a unique validation key
			$validation_key  = wp_generate_password( 64, false );
			$hashed		     = time() . ':' . wp_hash_password( $validation_key );

			$this->update_pending_status( $user_id, 2, $hashed );

			$to	     = get_option( 'admin_email' );
			$subject = sprintf( __( '[%s] Activate an administrator account' ), '###SITENAME###' );
			$message = __( 'A user has been designated as `administrator` on your website. Please visit the users page to approve or delete their account: ', 'secupress' ) . esc_url_raw( network_admin_url( 'users.php?secupress_users=pending' ) );

			secupress_send_mail( $to, $subject, $message );
		}
	}

	/**
	 * Prevent users with "pending" (2) user_status from logging in.
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (WP_User|WP_Error) $user WP_User object if authentication is successful, WP_Error object otherwise.
	 * 
	 * @return (WP_User|WP_Error) Modified WP_User object or WP_Error object.
	 */
	public function prevent_login_for_pending_users( $user ) {
		if ( secupress_is_user( $user ) && $this->is_pending_user( $user ) ) {
			$user = null; // Do not create WP_Error here, let WP display the default message "Error: Invalid username, email address or incorrect password." to be very vague.
		}

		return $user;
	}

	/**
	 * Prevent any modification of site_admins option
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 */
	public function do_not_grant_or_revoke_super_admin() {
		switch( current_action() ) {
			case 'grant_super_admin':  secupress_die( __( 'Sorry, you are not allowed to grant `Super Admin` to this user.', 'secupress' ), '', [ 'response' => 403, 'force_die' => true, 'attack_type' => 'users' ] ); break;
			case 'revoke_super_admin': secupress_die( __( 'Sorry, you are not allowed to revoke `Super Admin` to this user.', 'secupress' ), '', [ 'response' => 403, 'force_die' => true, 'attack_type' => 'users' ] ); break;
			default: secupress_die( __( 'Sorry, you are not allowed to edit this user.', 'secupress' ), '', [ 'response' => 403, 'force_die' => true, 'attack_type' => 'users' ] ); break;// Should not happen
		}
	}

	/**
	 * Force the logout for a user that is fake or pending
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (array) $dummy
	 * @param (WP_User) $user
	 */
	public function force_disconnect_fake_pending_user( $dummy, $user ) {
		if ( secupress_is_fake_user( $user->ID ) || $this->is_pending_user( $user ) ) {
			wp_destroy_current_session();
			wp_safe_redirect( add_query_arg( 'loggedout', '1', wp_login_url( '', true ) ) );
			die();
		}
	}

	/**
	 * Execute SQL query on module activation.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function activate_user_protection() {
		global $wpdb;

		$this->deactivate_user_protection( true );
		
		$seed       = rand( 1009, 99989 );
		secupress_set_option( 'secupress_user_protection_seed', $seed );
		$modulo     = rand( 100003, 9999973 );
		$modulo     = secupress_next_prime( $modulo );
		secupress_set_option( 'secupress_user_protection_modulo', $modulo );
		
		$query      = $wpdb->prepare( 
			"INSERT INTO $wpdb->usermeta (user_id, meta_key, meta_value) 
			SELECT ID, %s, (ID * %d) %% %d 
			FROM {$wpdb->users}",
			SECUPRESS_USER_PROTECTION,
			$seed,
			$modulo
		);
		
		$wpdb->query( $query );

		$wpdb->query( "ALTER TABLE $wpdb->users ADD UNIQUE INDEX unique_user_pass (user_pass)" );

		update_option( SECUPRESS_ADMIN_IDS, secupress_get_admin_ids_by_capa(), '', 'no' );
		if ( is_multisite() ) {
			update_site_option( SECUPRESS_SUPER_ADMINS, get_super_admins(), 'no' );
		}
	}

	/**
	 * Execute SQL query on module deactivation.
	 *
	 * @since 2.2.6
	 */
	public function deactivate_user_protection( $only_db = false ) {
		global $wpdb;

		if ( ! $only_db ) {
			secupress_set_option( 'secupress_user_protection_modulo', '' );
			secupress_set_option( 'secupress_user_protection_seed', '' );
			$index_exists = $wpdb->get_row(
				"SHOW INDEX FROM $wpdb->users WHERE Key_name = 'unique_user_pass'"
			);

			if ( ! $index_exists ) {
				$wpdb->query( "ALTER TABLE $wpdb->users ADD UNIQUE INDEX unique_user_pass (user_pass)" );
			}
		}

		$wpdb->query( $wpdb->prepare( "DELETE FROM $wpdb->usermeta WHERE meta_key = %s", SECUPRESS_USER_PROTECTION ) );
		
		delete_option( SECUPRESS_ADMIN_IDS );

		if ( is_multisite() ) {
			delete_site_option( SECUPRESS_SUPER_ADMINS );
		}
	}

	/**
	 * Give our saved option instead of real actual super admins
	 *
	 * @since 2.2.6
	 * @return (array)
	 */
	public function pre_site_option_site_admins() {
		return get_site_option( SECUPRESS_SUPER_ADMINS );
	}

	/**
	 * Highlight users without metadata in the users.php table.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function highlight_users_without_metadata() {
		global $pagenow;

		if ( 'users.php' !== $pagenow ) {
			return;
		}
		$fake_users   = secupress_get_fake_users();
		$user_row_ids = array_map( function( $user ) {
			return '#user-' . $user->ID;
		}, $fake_users );
		$user_row_ids = implode( ',', $user_row_ids );
		?>
		<style type="text/css">
			<?php echo $user_row_ids; ?> {
				background-image: linear-gradient(130deg, rgba(255, 170, 170, 0.2) 25%, transparent 25%, transparent 50%, rgba(255, 170, 170, 0.2) 50%, rgba(255, 170, 170, 0.2) 75%, transparent 75%, transparent 100%);
				background-size: 12px 15px;
			}
			.column-secupress_status {
				width: 5%;
			}
		</style>
		<?php
	}

	/**
	 * Prevent comments from posts where the author is a fake user.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param WP_Comment_Query $query The comment query object.
	 */
	public function exclude_comments_by_fake_users( $query ) {
		// Get the fake users
		$fake_users    = secupress_get_fake_users();
		$fake_user_ids = wp_list_pluck( $fake_users, 'ID' );

		// Exclude comments from posts where the author or post author is a fake user
		$query->query_vars[ 'author__not_in' ]      = $fake_user_ids;
		$query->query_vars[ 'post_author__not_in' ] = $fake_user_ids;
	}

	/**
	 * Exclude posts by fake users from wp_get_archives().
	 *
	 * @since 2.2.6
	 * @param string $where The SQL WHERE clause for the query.
	 * @return string The modified WHERE clause.
	 */
	public function exclude_archives_by_fake_users( $where ) {
		// Get the fake users
		$fake_users    = secupress_get_fake_users();
		$fake_user_ids = wp_list_pluck( $fake_users, 'ID' );

		// Exclude posts by fake users from archives
		$where .= " AND p.post_author NOT IN ( " . implode( ',', $fake_user_ids ) . " ) ";

		return $where;
	}

	/**
	 * Exclude terms associated with posts by fake users from get_terms().
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param array  $args    An array of get_terms() arguments.
	 * @param string $taxonomies The taxonomy or array of taxonomies.
	 * 
	 * @return array The modified get_terms() arguments.
	 */
	public function exclude_terms_by_fake_users( $args, $taxonomies ) {
		// Get the fake users
		$fake_users    = secupress_get_fake_users();
		$fake_user_ids = wp_list_pluck( $fake_users, 'ID' );

		// Exclude terms associated with posts by fake users
		$args['tax_query'][] = array(
		    'taxonomy' => $taxonomies,
		    'field'    => 'post_author',
		    'terms'    => $fake_user_ids,
		    'operator' => 'NOT IN',
		);

		return $args;
	}

	/**
	 * Filter users to show only those without metadata in "Fake Users" view.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param WP_Query $query The WP_Query instance.
	 */
	public function filter_fake_users( $query ) {
		global $pagenow;
		// Get the fake user IDs
		$f_user_ids = wp_list_pluck( secupress_get_fake_users(), 'ID' );
		$p_user_ids = secupress_get_pending_user_ids();
		if ( is_admin() && 'users.php' === $pagenow && isset( $_GET['secupress_users'] ) ) {
			switch ( $_GET['secupress_users'] ) {
				case 'fake':
					// Exclude the user IDs from the query
					$query->set( 'include', $f_user_ids );
				break;

				case 'pending':
					// Exclude the user IDs from the query
					$query->set( 'include', $p_user_ids );
				break;
			}
		}

		if ( ! is_admin() || ( is_admin() && 'users.php' === $pagenow ) ) {
			// Exclude the fake user IDs from the query
			$query->set( 'exclude', array_merge( $f_user_ids, $p_user_ids ) );
		}
	}

	/**
	 * Add the "Fake Users" view with the counter.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param array $views The user views.
	 * 
	 * @return array The modified user views.
	 */
	public function add_fake_and_pending_users_view( $views ) {
		// Get the users count
		$fake_users_count    = count( secupress_get_fake_users() );
		$pending_users_count = count( secupress_get_pending_user_ids() );

		// Add the "Fake Users" view with the counter
		if ( $pending_users_count ) {
			$pending_users_view = array(
				'secupress_pending_users' => '<a href="' . esc_url( add_query_arg( array( 'secupress_users' => 'pending' ), network_admin_url( 'users.php' ) ) ) . '">' . _n( 'Pending User', 'Pending Users', $pending_users_count, 'secupress' ) . '<span class="count"> (' . $pending_users_count . ')</span></a>',
			);

			// Insert the new view at the end of the views array
			$views = $views + $pending_users_view;

		}
		// Add the "Fake Users" view with the counter
		if ( $fake_users_count ) {
			$fake_users_view = array(
				'secupress_fake_users' => '<a href="' . esc_url( add_query_arg( array( 'secupress_users' => 'fake' ), network_admin_url( 'users.php' ) ) ) . '">' . _n( 'Fake User', 'Fake Users', $fake_users_count, 'secupress' ) . '<span class="count"> (' . $fake_users_count . ')</span></a>',
			);

			// Insert the new view at the end of the views array
			$views = $views + $fake_users_view;

		}

		if ( ! isset( $_GET['secupress_users'] ) ) {
			return $views;
		}
		// Links are clicked
		switch( $_GET['secupress_users'] ) {
			case 'fake':
				// Mark the "Fake Users" tab as selected
				foreach ( $views as $key => $value ) {
					if ( strpos( $key, 'secupress_fake_users' ) !== false ) {
						$views[ $key ] = str_replace( '<a ', '<a class="current" ', $value );
					} else {
						$views[ $key ] = str_replace( 'class="current" ', '', $value );
					}
				}
			break;
			case 'pending':
				// Mark the "Pending Users" tab as selected
				foreach ( $views as $key => $value ) {
					if ( strpos( $key, 'secupress_pending_users' ) !== false ) {
						$views[ $key ] = str_replace( '<a ', '<a class="current" ', $value );
					} else {
						$views[ $key ] = str_replace( 'class="current" ', '', $value );
					}
				}
			break;
		}

		return $views;
	}

	/**
	 * Count the number of users.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param array $result   
	 * @param string $strategy
	 * @param int $site_id 
	 * 
	 * @return (array)
	 */
	public function count_users( $result, $strategy, $site_id ) {
		global $wpdb, $pagenow;

		if ( 'users.php' !== $pagenow ) {
			return $result;
		}

		if ( ! $site_id ) {
			$site_id = get_current_blog_id();
		}
		remove_action( 'pre_get_users', [ $this, 'filter_fake_users' ] );
		$blog_prefix     = $wpdb->get_blog_prefix( $site_id );
		$result          = [];
		if ( $strategy === 'time' ) {
			$avail_roles = wp_roles()->get_names();

			$select_count   = [];
			foreach ( $avail_roles as $this_role => $name ) {
				$select_count[] = $wpdb->prepare( 'COUNT(NULLIF(`meta_value` LIKE %s, false))', '%' . $wpdb->esc_like( '"' . $this_role . '"' ) . '%' );
			}
			$select_count[] = "COUNT(NULLIF(`meta_value` = 'a:0:{}', false))";
			$select_count   = implode( ', ', $select_count );

			$row = $wpdb->get_row(
				"
				SELECT {$select_count}, COUNT(*)
				FROM {$wpdb->usermeta}
				INNER JOIN {$wpdb->users} ON user_id = ID
				WHERE meta_key = '{$blog_prefix}capabilities'
			",
				ARRAY_N
			);

			$col		   = 0;
			$role_counts   = array();
			foreach ( $avail_roles as $this_role => $name ) {
				$count = (int) $row[ $col++ ];
				if ( $count > 0 ) {
					$role_counts[ $this_role ] = $count;
				}
			}

			$role_counts['none'] = (int) $row[ $col++ ];

			// Subtract the fake and pending users count from role counts
			$fake_users     = wp_list_pluck( secupress_get_fake_users(), 'ID' );
			$pending_users  = secupress_get_pending_user_ids();
			$_users         = array_merge( $fake_users, $pending_users );
			$_users         = array_map( 'get_user_by', array_fill( 0, count( $_users ), 'ID' ), $_users );
			foreach ( $_users as $_user ) {
				$user_roles = secupress_get_user_roles( $_user->ID );
				foreach ( $user_roles as $role ) {
					if ( isset( $role_counts[ $role ] ) ) {
						$role_counts[ $role ] = max( 0, ( (int) $role_counts[ $role ] ) - 1 );
					}
				}
			}
			$total_users = (int) $row[ $col ] - count( $_users );

			$result['total_users'] = $total_users;
			$result['avail_roles'] = $role_counts;
		} else {
			$avail_roles = [
				'none' => 0,
			];

			$users_of_blog = $wpdb->get_col(
				"
				SELECT meta_value
				FROM {$wpdb->usermeta}
				INNER JOIN {$wpdb->users} ON user_id = ID
				WHERE meta_key = '{$blog_prefix}capabilities'
			"
			);

			foreach ( $users_of_blog as $caps_meta ) {
				$b_roles = maybe_unserialize( $caps_meta );
				if ( ! is_array( $b_roles ) ) {
				    continue;
				}
				if ( empty( $b_roles ) ) {
					$avail_roles['none']++;
				}
				foreach ( $b_roles as $b_role => $val ) {
					if ( isset( $avail_roles[ $b_role ] ) ) {
						$avail_roles[ $b_role ]++;
					} else {
						$avail_roles[ $b_role ] = 1;
					}
				}
			}

			// Subtract the fake and pending users count from role counts
			$fake_users     = wp_list_pluck( secupress_get_fake_users(), 'ID' );
			$pending_users  = secupress_get_pending_user_ids();
			$_users         = array_merge( $fake_users, $pending_users );
			$_users         = array_map( 'get_user_by', array_fill( 0, count( $_users ), 'ID' ), $_users );
			foreach ( $_users as $_user ) {
				$user_roles = secupress_get_user_roles( $_user->ID );
				foreach ( $user_roles as $role ) {
					if ( isset( $role_counts[ $role ] ) ) {
						$role_counts[ $role ] = max( 0, ( (int) $role_counts[ $role ] ) - 1 );
					}
				}
			}
			$total_users = (int) $row[ $col ] - count( $_users );

			$result['total_users'] = count( $users_of_blog ) - $_users;
			$result['avail_roles'] = $avail_roles;
		}

		return $result;
	}

	/**
	 * Add counter for fake users to "All users" menu entry.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function add_fake_users_counter() {
		global $menu;

		$fake_users = secupress_get_fake_users();
		if ( empty( $fake_users ) ) {
			return;
		}
		$fake_users_count = count( $fake_users );

		foreach ( $menu as $key => $item ) {
			if ( $item[2] === 'users.php' ) {
				$menu[ $key ][0] .= ' <span class="update-plugins count-1"><span class="plugin-count" title="' . esc_attr( __( 'Fake Users', 'secupress' ) ) . '">' . $fake_users_count . '</span></span>';
				break;
			}
		}
	}

	/**
	 * Filter fake user actions.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param array   $actions    An array of action links to be displayed.
	 * @param WP_User $user_object WP_User object.
	 * 
	 * @return array The modified action links array.
	 */
	public function add_user_status_action_link( $actions, $user_object ) {
		$fake_ids = secupress_get_fake_users();
		$fake_ids = wp_list_pluck( $fake_ids, 'ID' );

		if ( in_array( $user_object->ID, $fake_ids ) ) {
			if ( isset( $actions['delete'] ) ) {
				$actions = ['delete' => $actions['delete'] ];
			} else {
				$actions = [];
			}
		}

		$pend_ids = secupress_get_pending_user_ids();

		if ( in_array( $user_object->ID, $pend_ids ) ) {
			$_actions = [];

			$act1     = 'secupress_activate';
			$url1     = wp_nonce_url( admin_url( 'admin-post.php?user_id=' . $user_object->ID . '&action=' . $act1 ),  $act1 . '-' . $user_object->ID );
			$user_role         = esc_html( wp_sprintf( '%l', array_map( 'secupress_translate_user_role', $user_object->roles ) ) );
			if ( $user_role ) {
				$_actions[ $act1 ] = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url1 ), sprintf( _x( 'Activate as `%s`', 'user role', 'secupress' ), $user_role ) );
			} else {
				$_actions[ $act1 ] = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url1 ), _x( 'Activate without role', 'verb', 'secupress' ) );
			}

			$act2     = 'secupress_downgrade';
			$key2     = $act2 . ' delete';
			$url2     = wp_nonce_url( admin_url( 'admin-post.php?user_id=' . $user_object->ID . '&action=' . $act2 ), $act2 . '-' . $user_object->ID );
			// Only one identical role, do not set 2 links for the same action
			if ( $user_role !== get_option( 'default_role' ) ) {
				if ( $user_role ) {
					$_actions[ $key2 ]  = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url2 ), sprintf( _x( 'Downgrade as `%s`', 'user role', 'secupress' ), secupress_translate_user_role( get_option( 'default_role' ) ) ) );
			} else {
					$_actions[ $act2 ] = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url2 ), sprintf( _x( 'Activate as `%s`', 'user role', 'secupress' ), secupress_translate_user_role( get_option( 'default_role' ) ) ) );
				}
			}
			
			$act3     = 'secupress_downgrade&role=no';
			$key3     = $act3 . ' delete';
			$url3     = wp_nonce_url( admin_url( 'admin-post.php?user_id=' . $user_object->ID . '&action=' . $act3 ), $act2 . '-' . $user_object->ID ); // $act2 for nonce, ok!
			// Only if it's not already no role
			if ( $user_role !== '' ) {
				$_actions[ $key3 ]  = sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url3 ), sprintf( _x( 'Downgrade as `%s`', 'user role', 'secupress' ), __( 'No role', 'secupress' ) ) );
			}
			

			if ( isset( $actions['delete'] ) ) {
				$_actions['delete'] = $actions['delete'];
			}
			$actions = $_actions;
		}

		return $actions;
	}

	/**
	 * Activate a pending user
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 */
	public function activate_pending_user() {
		global $wpdb;

		$pending_user_ids = secupress_get_pending_user_ids();
		if ( ! isset( $_GET['user_id'], $_GET['_wpnonce'] ) || ! in_array( $_GET['user_id'], $pending_user_ids ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'secupress_activate-' . $_GET['user_id'] )) {
			wp_safe_redirect( remove_query_arg( 'secupress_users', wp_get_referer() ) );
			die();
		}
		$this->update_pending_status( $_GET['user_id'], 0 );
		
		update_option( SECUPRESS_ADMIN_IDS, secupress_get_admin_ids_by_capa(), 'no' );
		if ( is_admin() ) {
			secupress_add_transient_notice( __( 'User updated', 'secupress' ), 'updated' );
		}
		
		wp_safe_redirect( remove_query_arg( 'secupress_users', wp_get_referer() ) );
		die();
	}

	/**
	 * Downgrade a pending user
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 */
	public function downgrade_pending_user() {
		global $wpdb;

		$pending_user_ids = secupress_get_pending_user_ids();
		if ( ! isset( $_GET['user_id'], $_GET['_wpnonce'] ) || ! in_array( $_GET['user_id'], $pending_user_ids ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'secupress_downgrade-' . $_GET['user_id'] )) {
			wp_safe_redirect( remove_query_arg( 'secupress_users', wp_get_referer() ) );
			die();
		}
		$user_id  = $_GET['user_id'];
		$user     = new WP_User( $user_id );
		if ( secupress_is_user( $user ) ) {
			$role = isset( $GET['role'] ) && 'no' === $_GET['role'] ? false : get_option( 'default_role' ); // false = "no role"
			remove_action( 'set_user_role',    [ $this, 'update_user_and_send_validation_email' ], 10, 2 );
			remove_action( 'remove_user_role', [ $this, 'update_admin_ids_on_remove_user_role' ], 10, 2 );
			$user->set_role( $role );
			$this->update_pending_status( $user_id, 0 );

			secupress_add_transient_notice( __( 'User updated', 'secupress' ), 'updated' );
		}
		
		wp_safe_redirect( remove_query_arg( 'secupress_users', wp_get_referer() ) );
		die();
	}

	/**
	 * Add custom column to users.php table.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * @param array $columns The user table columns.
	 * 
	 * @return array The modified user table columns.
	 */
	public function add_status_column( $columns ) {
		global $pagenow;

		if ( 'users.php' !== $pagenow || ! isset( $_GET['secupress_users'] ) || 'fake' !== $_GET['secupress_users'] ) {
			return $columns;
		} 

		unset( $columns['role'] );
		$columns['secupress_up_status_roles']  = __( 'Role', 'secupress' );
		$columns['secupress_up_status_reason'] = __( 'Reason', 'secupress' );
		
		return $columns;
	}

	/**
	 * Prevent login without metadata.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * @param WP_User|null $user     The WP_User object if the user is authenticated successfully, null otherwise.
	 * @return WP_User|null The WP_User object if the user is authenticated successfully, null otherwise.
	 */
	public function prevent_login_without_metadata( $user ) {
		if ( secupress_is_user( $user ) && secupress_is_fake_user( $user->ID ) ) {
			$user = null; // Do not create WP_Error here, let WP display the default message "Error: Invalid username, email address or incorrect password." to be very vague.
		}
		return $user;
	}

	/**
	 * Prevent fake users to reset their password
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param WP_User $user_data The user data.
	 * 
	 * @return WP_User|null $user_data The user data.
	 */
	public function prevent_fake_users_password_reset( $user_data ) {
		$fake_users    = secupress_get_fake_users();
		$fake_user_ids = array_flip( wp_list_pluck($fake_users, 'ID') );

		// Prevent password reset for fake users
		if ( isset( $fake_user_ids[ $user_data->ID ] ) ) {
			return null;
		}
		return $user_data;
	}

	/**
	 * Populate custom column with content.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param string $value		The default column value.
	 * @param string $column_name  The column name.
	 * @param int    $user_id      The user ID.
	 * 
	 * @return string The modified column value.
	 */
	public function status_column_content( $value, $column_name, $user_id ) {
		if ( 'secupress_up_status_roles' === $column_name ) {
			return esc_html( wp_sprintf( '%l', array_map( 'secupress_translate_user_role', get_user( $user_id )->roles ) ) );
		}
		if ( 'secupress_up_status_reason' === $column_name ) {
			$style = 'color: rgba(255, 70, 70, 0.9)';

			switch ( secupress_is_fake_user( $user_id ) ) {

				case 'not_exists':
					$icon = 'hidden';
					$text = _x( 'Not Exist', 'the user', 'secupress' );
				break;

				case 'not_admin':
					$icon = 'businessman';
					$text = _x( 'Not Administrator', 'the user', 'secupress' );
				break;

				case 'wrong_passwordhash':
					$icon = 'admin-network';
					$text = __( 'Too Short Pass Hash', 'secupress' );
				break;
				
				case 'no_date':
					$icon = 'calendar-alt';
					$text = __( 'No Registration Date', 'secupress' );
				break;
				
				case 'no_nicename':
					$icon = 'nametag';
					$text = __( 'Missing user_nicename', 'secupress' );
				break;
				
				case 'no_metadata':
					$icon = 'dismiss';
					$text = __( 'Missing Metadata', 'secupress' );
				break;
				
				case 'no_modulo':
					$icon = 'database-remove';
					$text = __( 'Wrong Metadata Value', 'secupress' );
				break;
				
				case 'same_email':
					$icon = 'email';
					$text = __( 'Same Email Domain', 'secupress' );
				break;
				
				case 'wrong_email_mx':
					$icon = 'email-alt';
					$text = __( 'Wrong Email MX', 'secupress' );
				break;
				
				case 'wrong_email_dom':
					$icon = 'email-alt2';
					$text = __( 'Wrong Email Address', 'secupress' );
				break;
				
				default:
					$style = 'color: rgba(70, 70, 255, 0.8)';
					$icon  = 'yes-alt';
					$text  = '';
				break;
			}

			return sprintf( '<span class="dashicons dashicons-%s" style="%s"></span> %s', $icon, $style, esc_html( $text ) );
		}

	}

	/**
	 * Exclude posts authored by fake users.
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 * 
	 * @param WP_Query $query The WP_Query instance.
	 */
	public function exclude_posts_by_fake_users( $query ) {
		// Check if it's the main query and not in the admin area
		if ( ! is_admin() ) {
			// Get the fake users
			$fake_users    = secupress_get_fake_users();
			$fake_user_ids = wp_list_pluck( $fake_users, 'ID' );

			// Exclude posts authored by fake users
			$query->set( 'author__not_in', $fake_user_ids );
		}
	}

	/**
	 * Returns pending users (user_status=2) from custom query to prevent any filter
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @return (array)
	 */
	public function get_pending_user_ids() {
		global $wpdb;
		static $ids;

		if ( isset( $ids ) ) {
			return $ids;
		}

		$query = "SELECT ID FROM $wpdb->users WHERE user_status = 2";
		$ids   = $wpdb->get_col( $query );

		return $ids;
	}

	/**
	 * Returns if the given user, user_id is pending
	 *
	 * @since 2.2.6
	 * @author Julio Potier
	 * 
	 * @param (WP_User|int)
	 * @return (array)
	 */
	public function is_pending_user( $user ) {
		$user             = secupress_get_user_by( $user );
		$pending_user_ids = $this->get_pending_user_ids();

		return in_array( $user->ID, $pending_user_ids );
	}

	/**
	 * Remove bulk select input in users.php page
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function remove_bulk_select( $actions ) {
		if ( ! isset( $_GET['secupress_users'] ) ) {
			return $actions;
		}
		if ( isset( $actions['delete'] ) ) {
			$actions = [ 'delete' => $actions['delete'] ];
		} else {
			$actions = [];
		}
		return $actions;
	}
	
	/**
	 * Remove Change role select input in users.php page
	 *
	 * @author Julio Potier
	 * @since 2.2.6
	 */
	public function remove_change_role_select( $caps, $cap ) {
		if ( ! isset( $_GET['secupress_users'] ) ) {
			return $caps;
		}
		if ( 0 === strpos( $cap, 'promote_user' ) ) {
			return ['do_not_allow'];
		}

		return $caps;
	}

	
}

$GLOBALS['SecuPress_User_Protection'] = new SecuPress_User_Protection();
